/*******************************************************************/
/*                                                                 */
/*                      ADOBE CONFIDENTIAL                         */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                     */
/*                                                                 */
/* Copyright (c) 1986 - 2000 Adobe Systems Incorporated            */
/* All Rights Reserved.                                            */
/*                                                                 */
/* NOTICE:  All information contained herein is, and remains the   */
/* property of Adobe Systems Incorporated and its suppliers, if    */
/* any.  The intellectual and technical concepts contained         */
/* herein are proprietary to Adobe Systems Incorporated and its    */
/* suppliers and may be covered by U.S. and Foreign Patents,       */
/* patents in process, and are protected by trade secret or        */
/* copyright law.  Dissemination of this information or            */
/* reproduction of this material is strictly forbidden unless      */
/* prior written permission is obtained from Adobe Systems         */
/* Incorporated.                                                   */
/*                                                                 */
/*******************************************************************/

#ifdef MACINTOSH
#include <memory.h>         /* For FreeMem check below */
#endif
#include "fm_sgml.h"
#include "fstrings.h"
#include "fstrlist.h"
#include "fmemory.h"
#include "futils.h"
#include "fcharmap.h"

/*											   
 * application specific data
 */

/* 
 * a linked list of elements related to an IndexTerm element
 */
typedef struct index_entry {
        StringT name;
        StringT text;
        struct  index_entry *next;
} IndexEntryT;

typedef struct
{
	BoolT removeLeadingSpaces;
	IntT  inScreen;
	IntT  inLiteralLayout;
	F_ObjHandleT indexTermObjId; /* ID of index marker being processed */
	IndexEntryT *head;			 /* head of the linked list */
	IndexEntryT *current;		 /* current node in the linked list */
} SrAppDataT;

/*
 * function prototypes
 */
static IndexEntryT 	*newIndexEntry FARGS( (VoidT) );
static StringT     	writeIndexText FARGS( (IndexEntryT *p) );
static VoidT      	releaseIndex FARGS( (IndexEntryT *p) );
static VoidT 		validateListType FARGS((SrEventT *eventp, SrConvObjT srObj, StringT attrName));
static BoolT  		stripLeadingSpaces FARGS((SrConvObjT srObj));
static VoidT 		compressWhiteSpaces FARGS((SrConvObjT srObj, SrAppDataT *datap));
static VoidT 		createULinkTextInset FARGS((SrEventT *eventp, SrConvObjT srObj));
static BoolT 		isSpaceChar FARGS((UIntT c));
static BoolT 		isElementInlineGroup FARGS( (SrEventT *eventp, SrConvObjT srObj));
static StringT 		getCharTagFromElementName FARGS( (F_ObjHandleT docId, StringT etag) );
static StringT 		getCharBlock FARGS( (StringT tag) );
static F_ObjHandleT	getIdOfStartRangeMarker FARGS((F_ObjHandleT docId, StringT attrName, StringT val) );
static StringT		getAttrVal FARGS((F_ObjHandleT  docId, F_ObjHandleT elemId, StringT attrName));
static VoidT		appendIndexEntry FARGS((SrAppDataT *app, IndexEntryT *entryp));

/* macros */
#define NUM_ELEMENTS(x) (sizeof(x)/sizeof((x)[0]))


/*
 * SGML Reader Event Handler for DOCBOOK
 */
SrwErrorT Sr_EventHandler(eventp, srObj)
SrEventT *eventp;
SrConvObjT srObj;
{
	static SrAppDataT srdata;

	IndexEntryT *newIndex;
	
	BoolT removeLeadingSpaces;
	
	removeLeadingSpaces = srdata.removeLeadingSpaces;
	srdata.removeLeadingSpaces = False; 

	switch (eventp->evtype)
	{
	case SR_EVT_BEGIN_READER:
	case SR_EVT_BEGIN_BOOK:
	case SR_EVT_BEGIN_DOC:
	case SR_EVT_BEGIN_BOOK_COMP:
#ifdef MACINTOSH
	{
        static IntT mem = 0;
        if (!mem)
			mem = FreeMem();

        if (mem < 40000000)      /* value obtained from tests results */
			return(SRW_E_NOT_ENOUGH_MEM);
	}
#endif
		/*
		 * re-initialize data at the beginning of a
		 * session/book/document/book-component.
		 */
		F_ClearPtr(&srdata, sizeof(srdata));
		break;

	case SR_EVT_BEGIN_ELEM:
		/* "ItemizedList" and "ListItem" attributes need special
		 * validation for the "Mark" and "Override" attributes
		 * respectively.
		 */
		if (F_StrIEqual(eventp->u.tag.gi, (StringT)"ItemizedList"))
			validateListType(eventp, srObj, (StringT)"Mark");
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"ListItem"))
			validateListType(eventp, srObj, (StringT)"Override");

					
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"Para"))
		{
			/*
		     * Leading spaces in "Para" element should be removed.
		     */
			srdata.removeLeadingSpaces = True;
		
		}
		/*
		 * if we have a Title element within a Table and we are using
		 * the table title model, then change the Frame element tag
		 * to TableTitle instead of Title
		 */
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"Title"))
		{
			SrConvObjT parConvObj;
			StringT parTag;
			F_ObjHandleT docId, tableTitleElemDefId = 0;

			parConvObj = Sr_GetParentConvObj(srObj);
			parTag = Sr_GetFmElemTag(parConvObj);
			if (F_StrIEqual(parTag, (StringT)"Table"))
			{
				docId = Sr_GetDocId(srObj);
				tableTitleElemDefId = F_ApiGetNamedObject(
					docId, FO_ElementDef, (StringT)"TableTitle");
				if (tableTitleElemDefId)
					Sr_SetFmElemTag(srObj, (StringT)"TableTitle");
			}
			F_StrFree(parTag);
		}
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"Screen"))
			srdata.inScreen++;
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"LiteralLayout"))
			srdata.inLiteralLayout++;
		/*
		 * A "ULink" element should be mapped to a a text inset.
		 */
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"ULink"))
		{
			createULinkTextInset(eventp, srObj);
			return(SRW_E_SUCCESS);
		}

		/* 
		 * if the role attribute of Graphic or InlineGraphic is not
		 * set to "ImportedGraphic", "AnchoredFrame" or "AsIs" then
		 * set it to "AsIs"
		 */
		else if ( F_StrIEqual(eventp->u.tag.gi, (StringT)"Graphic")
				|| F_StrIEqual(eventp->u.tag.gi, (StringT)"InlineGraphic") )
		{
			StringT val;
			SrwErrorT err;
			F_ObjHandleT elemId;

			/* 
			 * convert the element
			 */
			err = Sr_Convert(eventp, srObj);
			elemId = Sr_GetFmElemId(srObj);

			/*
			 * get the value of the role attribute 
			 */
			val = getAttrVal(Sr_GetDocId(srObj), elemId, (StringT)"ROLE");

			/*
			 * if the value is not ImportGraphic, AnchoredFrame, AsIs
			 * then set it to AsIs
			 */
			if (!F_StrIsEmpty(val)
					&& !F_StrIEqual(val, (StringT)"ImportedGraphic")
			 		&& !F_StrIEqual(val, (StringT)"AnchoredFrame")
			 		&& !F_StrIEqual(val, (StringT)"AsIs") )
			{
				F_AttributesExT attrs;

				/*
				 * create and set the atrribute Role to AsIs
				 */
				attrs.len = 1;
				attrs.val = (F_AttributeExT *) F_Calloc(1, sizeof(F_AttributeExT), DSE);
				attrs.val[0].name = F_StrCopyString((StringT)"Role");
				attrs.val[0].values.len = 1;
				attrs.val[0].values.val = (StringT *) F_Alloc(sizeof(StringT), DSE);
				attrs.val[0].values.val[0] = F_StrCopyString((StringT)"AsIs");
				F_ApiSetAttributesEx(Sr_GetDocId(srObj), elemId, &attrs);	
				F_ApiDeallocateAttributesEx(&attrs);
			}

			F_StrFree(val);
			return err;
		}
			
		/*
		 * if we are processing an Inline Group element in an indexterm element 
		 * we need to save information about it. Otherwise, use the
		 * default processing
		 */
		else if (srdata.indexTermObjId && isElementInlineGroup(eventp, srObj))
		{
			StringT charfmt;
			F_ObjHandleT templId = Sr_GetTemplateDocId();

			newIndex = newIndexEntry();
			newIndex->name = F_StrCopyString(eventp->u.tag.gi);
			charfmt = getCharTagFromElementName(templId, Sr_GetFmElemTag(srObj)); 
			newIndex->text = F_StrCopyString(getCharBlock(charfmt));
			appendIndexEntry(&srdata, newIndex);

			F_StrFree(charfmt);
			return (SRW_E_SUCCESS);
		}
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"IndexTerm"))
		{
			SrConvObjT docObj;
			F_ObjHandleT docId;
			StringT pagenum_val = NULL; 
			StringT spanend_val = NULL;
			UIntT i;
			
			docObj = Sr_GetCurConvObjOfType(SR_OBJ_DOC);
			docId = Sr_GetDocId(docObj); 

			Sr_Convert(eventp, srObj);
			srdata.indexTermObjId = Sr_GetFmObjId(srObj);
		
			/* 
			 * Check if the PageNum attribute is set to zero, if yes,
			 * then a <$nopage> block will be inserted if the span end
			 * attribute is set then it signals the end  of the page
			 * range. 
			 */
			for (i=0; i<eventp->u.tag.sgmlAttrVals.len; i++)
			{
				SgmlAttrValT *sgmlAttrValp = &(eventp->u.tag.sgmlAttrVals.val[i]);
				if (F_StrIEqual(sgmlAttrValp->sgmlAttrName, (StringT)"PAGENUM"))
				{
					pagenum_val =
						F_StrCopyString(F_StrListGet(sgmlAttrValp->sgmlAttrVal, 0));
				}
				else if (F_StrIEqual(sgmlAttrValp->sgmlAttrName, (StringT)"SPANEND"))
				{
					spanend_val =
						F_StrCopyString(F_StrListGet(sgmlAttrValp->sgmlAttrVal, 0));
				}		
			}		
				
			newIndex = newIndexEntry();
			newIndex->name = F_StrCopyString(eventp->u.tag.gi);
			
			/* 
			 * if there is a spanend attribute then we place the <$endrange>
			 * building block and ignore the PageNum attribute because Frame
			 * cannot handle both <$endrange> and <$nopage>
			 */
			if (!F_StrIsEmpty(spanend_val))
			{
				F_ObjHandleT markerId;

				/* 
				 *  still need to set the <$startrange> block in the element
				 *  whose ID is specified here
				 */
				 markerId = getIdOfStartRangeMarker(docId, (StringT)"ID", spanend_val);
				 if (markerId)
				 {
				 	StringT text, newText;

				 	text = F_ApiGetString(docId, markerId, FP_MarkerText);
					newText = F_StrNew(F_StrLen(text) + 16);
					F_Sprintf(newText, "<$startrange>%s", text);
					F_ApiSetString(docId, markerId, FP_MarkerText, newText);
					F_Sprintf(newText, "<$endrange>%s", text);
					newIndex->text = newText;
					F_StrFree(text);
				}
			}
			else if (F_StrEqual(pagenum_val, (StringT)"0"))
				newIndex->text = F_StrCopyString((StringT)"<$nopage>");

			appendIndexEntry(&srdata, newIndex);

			F_StrFree(pagenum_val);
			F_StrFree(spanend_val);
			return (SRW_E_SUCCESS);
		}
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"Primary"))
		{
			newIndex = newIndexEntry();
			newIndex->name = F_StrCopyString(eventp->u.tag.gi);
			newIndex->text = F_StrCopyString((StringT)"");
			appendIndexEntry(&srdata, newIndex);
			return(SRW_E_SUCCESS);
		}
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"Secondary"))
		{
			newIndex = newIndexEntry();
			newIndex->name = F_StrCopyString(eventp->u.tag.gi);
			newIndex->text = F_StrCopyString((StringT)":");
			appendIndexEntry(&srdata, newIndex);
			return(SRW_E_SUCCESS);
		}
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"Tertiary"))
		{
			newIndex = newIndexEntry();
			newIndex->name = F_StrCopyString(eventp->u.tag.gi);
			newIndex->text = F_StrCopyString((StringT)":");
			appendIndexEntry(&srdata, newIndex);
			return(SRW_E_SUCCESS);
		}
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"See"))
		{
			newIndex = newIndexEntry();
			newIndex->name = F_StrCopyString(eventp->u.tag.gi);
			newIndex->text = F_StrCopyString((StringT)"<IndexSee>See <Default Para Font>");
			appendIndexEntry(&srdata, newIndex);
			return(SRW_E_SUCCESS);
		}
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"SeeAlso"))
		{
			newIndex = newIndexEntry();
			newIndex->name = F_StrCopyString(eventp->u.tag.gi);
			newIndex->text = F_StrCopyString((StringT)"<IndexSee>See also <Default Para Font>");
			appendIndexEntry(&srdata, newIndex);
			return(SRW_E_SUCCESS);
		}
		break;
	case SR_EVT_END_ELEM:
		/*
		 * Spaces immediately after a "Screen" or a "LiteralLayout"
		 * element should be removed.
		 */
		if (F_StrIEqual(eventp->u.tag.gi, (StringT)"Screen"))
		{
			srdata.inScreen--;
			srdata.removeLeadingSpaces = True;
		}
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"LiteralLayout"))
		{
			srdata.inLiteralLayout--;
			srdata.removeLeadingSpaces = True;
		}
		/* 
		 * inline group element in indexterm
		 */
		else if (srdata.indexTermObjId && isElementInlineGroup(eventp, srObj) )
		{
	    	newIndex = newIndexEntry();
			newIndex->name = F_StrCopyString(eventp->u.tag.gi);
			/* 
			 * the end tag of an inline group element in an indexterm element
			 *  maps to <Default Para Font>
			 */
			newIndex->text = F_StrCopyString((StringT)"<Default Para Font>");
			appendIndexEntry(&srdata, newIndex);
			return (SRW_E_SUCCESS);
		}
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"IndexTerm"))
		{
			SrConvObjT docObj;
			F_ObjHandleT docId;
			StringT theIndexText;
			
			docObj = Sr_GetCurConvObjOfType(SR_OBJ_DOC);
			docId = Sr_GetDocId(docObj); 

			/* 
			 * set the marker text and set the falg to zero to exit
			 * the scope of an indexterm element
			 */
            theIndexText = writeIndexText(srdata.head);
            F_ApiSetString(docId, srdata.indexTermObjId, FP_MarkerText, theIndexText);
            F_ApiDeallocateString(&theIndexText);

			/* free linked list, reset
			 * pointers, etc.
			 */
			releaseIndex(srdata.head);
			srdata.head = NULL;
			srdata.current = NULL;
			srdata.indexTermObjId = 0;

			return(SRW_E_SUCCESS);	
		}
		else if (F_StrIEqual(eventp->u.tag.gi, (StringT)"Primary") ||
					F_StrIEqual(eventp->u.tag.gi, (StringT)"Secondary") ||
					F_StrIEqual(eventp->u.tag.gi, (StringT)"Tertiary") )
		{
			StringT val = NULL; 
			UIntT i;

			for (i=0; i < eventp->u.tag.sgmlAttrVals.len; i++)
			{
				SgmlAttrValT *sgmlAttrValp = &(eventp->u.tag.sgmlAttrVals.val[i]);
				if (F_StrIEqual(sgmlAttrValp->sgmlAttrName, (StringT)"SORTAS"))
				{
					val = F_StrCopyString(F_StrListGet(sgmlAttrValp->sgmlAttrVal, 0));
				break;
				}	
			}

			if (val)
			{
				StringT tmp = F_StrNew(F_StrLen(val) + 2);
			
				newIndex = newIndexEntry();
				newIndex->name = F_StrCopyString((StringT)"SortAs");
				F_Sprintf(tmp, "[%s]", val);
				newIndex->text = F_StrCopyString(tmp);
				appendIndexEntry(&srdata, newIndex);
				F_StrFree(val);
				F_StrFree(tmp);
			}
			
			return(SRW_E_SUCCESS);
		}
		break;

	
	case SR_EVT_CDATA:
	case SR_EVT_RE:

		/*
		 * Remove or compress white spaces in text as needed.
		 */
		if (srObj && Sr_GetObjType(srObj) == SR_OBJ_TEXT)
		{
			if (removeLeadingSpaces)
				srdata.removeLeadingSpaces = stripLeadingSpaces(srObj);
			compressWhiteSpaces(srObj, &srdata);

			/*
			 * text in an indexterm element needs to be saved
			 */
			if(srdata.indexTermObjId)
			{
				newIndex = newIndexEntry();
				newIndex->name = F_StrCopyString((StringT)"<TEXT>");
				newIndex->text = Sr_GetFmText(srObj);
				appendIndexEntry(&srdata, newIndex);
				return (SRW_E_SUCCESS);
			}

		}
		break;
	
	default:
		break;
	}

	return Sr_Convert(eventp, srObj);
}

/*
 * "ItemizedList" and "ListItem" attributes need special
 * validation for the "Mark" and "Override" attributes
 * respectively. This implementation of DOCBOOK handles
 * the following values for these attributes:
 *		- Bullet
 *		- Dash
 *		- Box
 *		- Check
 * All other values result in an error message.
 */
static VoidT validateListType(eventp, srObj, attrName)
SrEventT *eventp;
SrConvObjT srObj;
StringT attrName;
{
	F_AttributeExT attVal;
	StringT val = NULL;
	UIntT i;

	for (i=0; i<eventp->u.tag.sgmlAttrVals.len; i++)
	{
		SgmlAttrValT *sgmlAttrValp = &(eventp->u.tag.sgmlAttrVals.val[i]);
		if (F_StrIEqual(sgmlAttrValp->sgmlAttrName, attrName))
		{
			val = F_StrCopyString(F_StrListGet(sgmlAttrValp->sgmlAttrVal, 0));
			break;
		}
	}

	if (!val)
		return;

	F_StrStripLeadingSpaces(val);
	F_StrStripTrailingSpaces(val);

	if (F_StrIsEmpty(val))
	{
		F_StrFree(val);
		return;
	}

	F_ClearPtr(&attVal, sizeof(F_AttributeExT));
	attVal.name = attrName;
	attVal.values.len = 1;
	attVal.values.val = &val;

	if (F_StrIEqual(val, (StringT)"Bullet"))
		F_StrCpy(val, (StringT)"Bullet");
	else if (F_StrIEqual(val, (StringT)"Dash"))
		F_StrCpy(val, (StringT)"Dash");
	else if (F_StrIEqual(val, (StringT)"Box"))
		F_StrCpy(val, (StringT)"Box");
	else if (F_StrIEqual(val, (StringT)"Check"))
		F_StrCpy(val, (StringT)"Check");
	else
	{
		/*
		 * TODO: log an error message
		 */
	}

	Sr_SetAttrVal(srObj, &attVal);
	F_StrFree(val);
}

/*
 * Strips leading spaces. Returns "True" if the text string is empty or
 * if it contains spaces only. Returns "False" otherwise.
 */
static BoolT stripLeadingSpaces(srObj)
SrConvObjT srObj;
{
	StringT s, t;
	BoolT rc;

	s = Sr_GetFmText(srObj);
	if (F_StrIsEmpty(s))
		rc = True;
	else
	{
		t = s;
		while (*t && isSpaceChar(*t))
			t++;
		Sr_SetFmText(srObj, t);
		rc = F_StrIsEmpty(t);
	}

	F_StrFree(s);
	return(rc);
}

/*
 * Compress white spaces based on current context.
 */
static VoidT compressWhiteSpaces(srObj, datap)
SrConvObjT srObj;
SrAppDataT *datap;
{
	StringT s;
	IntT i, j;

	if (datap->inScreen || datap->inLiteralLayout)
		return;
	/*
	 * get text string from the object.
	 */
	s = Sr_GetFmText(srObj);
	if (F_StrIsEmpty(s))
	{
		F_StrFree(s);
		return;
	}

	/*
	 * compress multiple spaces
	 */
	for (i=j=0; s[i];)
	{
		s[j] = s[i];
		if (!s[i])
			break;

		if (isSpaceChar(s[i]))
		{
			while (s[i] && isSpaceChar(s[i]))
				i++;
		}
		else
			i++;
		j++;
	}
	s[j] = s[i];

	/*
	 * if last character in the string is a space
	 * character, we need to strip any leading space
	 * characters in any immediately following string.
	 */
	if (!F_StrIsEmpty(s) && isSpaceChar(s[F_StrLen(s) - 1]))
		datap->removeLeadingSpaces = True;

	/*
	 * post back the altered string to the text object.
	 */
	Sr_SetFmText(srObj, s);
	F_StrFree(s);
}

/*
 * create a text inset to the file referenced by the URL attribute.
 */
static VoidT createULinkTextInset(eventp, srObj)
SrEventT *eventp;
SrConvObjT srObj;
{
	F_PropValsT props, *retprops = NULL;
	FilePathT *fpSgml, *fp, *srcDirPath;
	StringT filename = NULL;
	F_ElementLocT eloc;
	F_TextLocT tloc;
	SrConvObjT parentObj, docObj;
	F_ObjHandleT docId;
	UIntT i;
	IntT err;
	BoolT smartQuotes;

	for (i=0; i<eventp->u.tag.sgmlAttrVals.len; i++)
	{
		if (F_StrIEqual(eventp->u.tag.sgmlAttrVals.val[i].sgmlAttrName,
						(StringT)"URL"))
		{
			filename = F_StrCopyString(F_StrListGet(
					eventp->u.tag.sgmlAttrVals.val[i].sgmlAttrVal, 0));
			break;
		}
	}

	if (!filename)
		return;

	docObj = Sr_GetCurConvObjOfType(SR_OBJ_DOC);
	if (!docObj)
	{
		docObj = Sr_GetCurConvObjOfType(SR_OBJ_BOOK_COMP);
		if (!docObj)
			goto done;
	}
	docId = Sr_GetDocId(docObj);

	parentObj = Sr_GetParentConvObj(srObj);
	while (parentObj && !(eloc.parentId = Sr_GetFmElemId(parentObj)))
		parentObj = Sr_GetParentConvObj(parentObj);
	if (!docId || !eloc.parentId)
		goto done;

	eloc.childId = 0;
	eloc.offset = 0;
	tloc = F_ApiElementLocToTextLoc(docId, &eloc);
	if (!tloc.objId)
		goto done;

	/*
	 * If filepath is relative, it is assumed to be present in the
	 * same directory as the SGML source document.
	 */
	fpSgml = Srw_GetStructuredDocFilePath();
	if (!fpSgml)
		goto done;
	srcDirPath = F_FilePathParent(fpSgml, &err);
	F_FilePathFree(fpSgml);
	if (!srcDirPath)
		goto done;

	fp = F_PathNameToFilePath(filename, srcDirPath, FDefaultPath);
	F_FilePathFree(srcDirPath);
	if (fp)
	{
		F_StrFree(filename);
		filename = F_FilePathToPathName(fp, FDefaultPath);
		F_FilePathFree(fp);
	}

	props = F_ApiGetImportDefaultParams();
	i = F_ApiGetPropIndex(&props, FS_ForceImportAsText);
	props.val[i].propVal.u.ival = True;
	smartQuotes = F_ApiGetInt(FV_SessionId, docId, FP_SmartQuotes);
	F_ApiSetInt(FV_SessionId, docId, FP_SmartQuotes, False);
	F_ApiImport(docId, &tloc, filename, &props, &retprops);
	F_ApiSetInt(FV_SessionId, docId, FP_SmartQuotes, smartQuotes);
	F_ApiDeallocatePropVals(&props);
	F_ApiDeallocatePropVals(retprops);

  done:
	F_StrFree(filename);
}

static BoolT isSpaceChar(c)
UIntT c;
{
	return(c == FC_SPACE || c == FC_TAB);
}
	
/*
 * allocates heap memory for an IndexEntryT. It initalizes
 * the members of the structure
 */

static IndexEntryT 
*newIndexEntry()
{
	return (IndexEntryT *) F_Calloc(1, sizeof(IndexEntryT), DSE);
}

/* 
 * create the Index Marker Text by traversing all the nodes
 */
static StringT
writeIndexText(p)
IndexEntryT *p;
{
	StringT text;
	StringT saveSortAs = NULL;
	BoolT fseealso = True;

	/* allocate enough memory to hold the string */
	text = F_StrNew(2000);

	/* traverse the linked list */
	while (p != NULL)
	{
		/* if there is a sortas, save its value for later */
		if (F_StrIEqual(p->name, (StringT)"SortAs") && p->next != NULL)
		{
			if (F_StrIEqual(p->next->name, (StringT)"See") ||
				F_StrIEqual(p->next->name, (StringT)"SeeAlso") )
					saveSortAs = F_StrCopyString(p->text);
			else
				F_StrCat(text, p->text);
		}
		/* separate multiple SeeAlso with a semicolon */
		else if (F_StrIEqual(p->name, (StringT)"SeeAlso") )
		{
			if (fseealso)  /* first seealso in the list */
			{
				F_StrCat(text, p->text);
				/* <IndexSee>See also <Default Para Font> */
				fseealso = False;
			}
			else  /* separate subsequent seealso with semicolons */
				F_StrCat(text, (StringT)"\\;");
		}
		else
			F_StrCat(text, p->text);

		p = p->next;
	}

	/* sortas is always at the end of the string */
	if (saveSortAs)
		F_StrCat(text, saveSortAs);	
	F_StrFree(saveSortAs);

	return text;
}

/* 
 * releases the heap memory allocated for the link list of IndexEntryT
 * elements
 */

static VoidT
releaseIndex(p)
IndexEntryT *p;
{
	IndexEntryT *temp;

  	while (p != NULL)
	{
		temp = p->next;
		F_StrFree(p->text);
		F_StrFree(p->name);
		F_Free(p);
		p = temp;
	}
}

/*
 * determine if the element being processed is part
 * of the InlineGroup entity. This application supports
 * only <keycap> but it can be modified easily to support
 * any other element in that group
 */
static BoolT
isElementInlineGroup(eventp, srObj)
SrEventT *eventp;
SrConvObjT srObj;
{

	StringT elemName = eventp->u.tag.gi;
	if (F_StrIEqual(elemName, (StringT)"KeyCap"))
		return True;
	else
		return False;
}

/*
 * If the element specified by the etag argument has a text range
 * character associated with it, this function returns the associated
 * character tag. Otherwise, it returns etag.
 */

static StringT 
getCharTagFromElementName(docId, etag)
F_ObjHandleT docId;
StringT etag;
{

	F_ObjHandleT fmtRulesId, fmtClauseId;
	F_ObjHandleT edefId; 
	StringT formatTag; /* needs to be freed */

	edefId = F_ApiGetNamedObject(docId, FO_ElementDef, etag);

	if (!edefId) 
		return etag;

	fmtRulesId = F_ApiGetId(docId, edefId, FP_TextFmtRules);

	/* use the element name if there is no character tag */
	if (!fmtRulesId)
		return etag;

	fmtClauseId = F_ApiGetId(docId, fmtRulesId, FP_FmtRuleClause);
	if (!fmtClauseId)
		return etag;

	/* otherwise return the first character tag */
	formatTag = F_ApiGetString(docId, fmtClauseId, FP_FormatTag);
	if (!formatTag)
		return etag;

	return formatTag;
}

/*
 * enclose the character tag in <> to produce the
 * building block
 */

static StringT 
getCharBlock(charTag)
StringT charTag;	
{
	StringT block;

	block = (StringT) F_Alloc(F_StrLen(charTag) + 3, NO_DSE);

	if (F_StrEqual(charTag, (StringT)""))
		F_StrCpy(block, (StringT)"");	
	else 
		F_Sprintf(block, "%c%s%c", '<', charTag, '>');
		
	return block;
}

/*
 * gets the ObjectId of the Index Marker that started the <$endrange>
 */
static F_ObjHandleT
getIdOfStartRangeMarker(docId, attrName, val)
F_ObjHandleT docId;
StringT attrName;
StringT val;
{
	F_ObjHandleT markerId, foundMarkerId = 0;

	/* loop through all the index markers in document */
	for (markerId = F_ApiGetId(0, docId, FP_FirstMarkerInDoc);
			markerId;
			markerId = F_ApiGetId(docId, markerId, FP_NextMarkerInDoc)
		)
	{
		F_AttributesExT attr;
		UIntT i;
		F_ObjHandleT elemId;

		elemId = F_ApiGetId(docId, markerId, FP_Element);
		if (elemId)
		{
			attr = F_ApiGetAttributesEx(docId, elemId);

			/* get the value of the atribute in attrName */
			for (i=0; i < attr.len; i++)
			{
				F_AttributeExT *attrValp = &(attr.val[i]);
				if (F_StrIEqual(attrValp->name, attrName) &&
					attrValp->values.len &&
					attrValp->values.val &&
					F_StrIEqual(val, attrValp->values.val[0]))
				{
					foundMarkerId = markerId;	
					break;
				}
			}

			F_ApiDeallocateAttributesEx(&attr);
			if (foundMarkerId)
				break;
		}
	}
	return foundMarkerId;
}

/*
 * Returns the attribute value of the attribute specified
 * by 'attrName' in the element specified by 'elemId'
 */
static StringT
getAttrVal(docId, elemId, attrName)
F_ObjHandleT docId, elemId;
StringT attrName;
{
	StringT val = NULL;
	F_AttributesExT attr;
	UIntT i;

	if (!docId || !elemId)
		return(NULL);

	attr = F_ApiGetAttributesEx(docId, elemId);

	/* get the value of the atribute in attrName */
	for (i=0; i < attr.len; i++)
	{
		F_AttributeExT *attrValp = &(attr.val[i]);
		if (F_StrIEqual(attrValp->name, attrName))
		{
			if (attrValp->values.len
				&& attrValp->values.val
				&& attrValp->values.val[0])
				val = F_StrCopyString(attrValp->values.val[0]);
			break;
		}
	}

	F_ApiDeallocateAttributesEx(&attr);
	return val;
}

/*
 * append new index entry to the list
 */
static VoidT
appendIndexEntry(SrAppDataT *app, IndexEntryT *entryp)
{
	entryp->next = NULL;
	if (app->current)
		app->current->next = entryp;
	else
		app->head = entryp;
	app->current = entryp;
}

