/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

// Define the types here
const typeTooltip = 'tooltip';
const typeTourCoachmark = 'tour_coachmark';

//////////////////////////////////////////////////////////////////////////////
//
// A TourCoachmarkDisplayAdapter object represents a single coachmark,
// a tour coachmark (following specs of Rush First Mile Experience) in particular
// (based on IDisplayAdapter)
//
function TourCoachmarkDisplayAdapter(/** String */ inType)
{
	var coachmark = null;
	var params = {};
	var content = null;

	var contentStrings = {};

	contentStrings.title 		= '';
	contentStrings.text 		= '';
	contentStrings.debugText 	= '';

	var contentLoaded = false;
	var visible = false;

	// true if the object was terminated via call this.terminate()
	// If the TourCoachmarkDisplayAdapter is requesting its content from the ContentProviderManager
	// and the ContentProviderManager in turn needs to read the according file then there is an
	// asynchronous behavior. I.e. TourCoachmarkDisplayAdapter can be termated between requesting the
	// content and receiving the content. If that happens and the requested content is received we need to
	// ignore and stop proceeding.
	var terminated = false;

    //////////////////////////////////////////////////////////////////////////////
    //
    // Return the type string of this adapter.
    // The type string of a WorkflowDisplay object need to match
    //
    this.getType = function()
    {
			if(inType == typeTooltip)
			{
				return TourCoachmarkDisplayAdapter.TourCoachmarkDisplayAdapter.TOOLTIP_TYPE_ID;
			}
			else if(inType == typeTourCoachmark)
			{
				return TourCoachmarkDisplayAdapter.TourCoachmarkDisplayAdapter.TYPE_ID;
			}
      return TourCoachmarkDisplayAdapter.TYPE_ID;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize adapter
    //
    this.initialize = function(/*Object*/ inParams)
    {
        params = Utils.getParamValue(inParams, Utils.REQUIRED, {});

		Utils.throwInvalid(params.ctrlPath);
		}

    //////////////////////////////////////////////////////////////////////////////
    //
    // Terminate adapter instance
    //
    this.terminate = function()
    {
		// remember termination state
		terminated = true;

		this.hide();
		// and also dereference
		coachmark = null;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // show coachmark
    //
	this.show = function()
	{
		if (contentLoaded)
		{
			doShow();
		}
		else
		{
			loadContent(true);
		}
	}

    //////////////////////////////////////////////////////////////////////////////
    //
    // Hide coachmark
    //
    this.hide = function()
    {
		if (Utils.isValidProperty(coachmark))
		{
			coachmark.hide();
			visible = false;
		}
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Display new content in coachmark
    //
    this.setContent = function(/*WorkflowContent*/ inContent)
    {
        content = inContent;
		contentLoaded = false;

		if (visible && Utils.isValidProperty(content))
		{
			this.hide();
			this.show();
		}
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Set or change existing parameters
	//
    this.setParameters = function(/*Object*/ inParams)
	{
		if (visible)
		{
			var refresh =	(Utils.isValidProperty(inParams.ctrlPath) &&
							(params.ctrlPath != inParams.ctrlPath)) ||
							(Utils.isValidProperty(inParams.anchor) &&
							Utils.isValidProperty(params.anchor) &&
							(params.anchor != inParams.anchor));

			var enabledState =	Utils.isValidProperty(inParams.enabledButton) &&
								Utils.isValidProperty(params.enabledButton) &&
								(inParams.enabledButton != params.enabledButton);

			// store new params
			//
			for (var p in inParams)
			{
				if (inParams.hasOwnProperty(p))
				{
					params[p] = inParams[p];
				}
			}

			if (refresh)
			{
				this.hide();
				this.show();
			}
			else if (enabledState)
			{
				var enabled = true;

				if (Utils.isValidProperty(params.enabledButton, Boolean))
				{
					enabled = params.enabledButton;
				}

				coachmark.setEnableNext(enabled);
			}
		}
		else
		{
			params = inParams;
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// show coachmark
	//
	function doShow()
	{
		if (!Utils.isValidProperty(coachmark))
		{
			var showNext 			  = "";
			var nextBtnText			  = "";
			var anchor				  = "";
			var ctrlPath 			  = "";
			var trackVisibility		  = true;
			var bypassVisibilityCheck = false;
			var displayAsTooltip	  = false;

			if(inType == typeTooltip)
			{
                anchor                 = Utils.getParamValue(params.anchor, Utils.OPTIONAL, Coachmark.ANCHOR_BOTTOM_EDGE);
                ctrlPath               = Utils.getParamValue(params.ctrlPath, Utils.REQUIRED);
                trackVisibility        = Utils.getParamValue(params.trackVisibility, Utils.OPTIONAL, true);
                displayAsTooltip       = true;
			}
			else if(inType == typeTourCoachmark)
			{
                showNext               = Utils.getParamValue(params.showNext, Utils.OPTIONAL, "");
                nextBtnText            = Utils.getParamValue(params.nextBtnText, Utils.OPTIONAL, "");
                anchor                 = Utils.getParamValue(params.anchor, Utils.OPTIONAL, Coachmark.ANCHOR_BOTTOM_EDGE);
                ctrlPath               = Utils.getParamValue(params.ctrlPath, Utils.REQUIRED);
                trackVisibility        = Utils.getParamValue(params.trackVisibility, Utils.OPTIONAL, true);
                bypassVisibilityCheck  = Utils.getParamValue(params.bypassVisibilityCheck, Utils.OPTIONAL, false);
                displayAsTooltip       = false;
			}
            
			ctrlPath = MonikerMapper.get().getValue(ctrlPath);
			anchor = MonikerMapper.get().getValue(anchor);

			var cm = null;
			try
			{
				var coachParams = new Object;
				coachParams.nodeUniqueID = ctrlPath;
				coachParams.anchor = anchor;
				coachParams.title = contentStrings.title;
				coachParams.text = contentStrings.text;
				coachParams.debugText = contentStrings.debugText;
				coachParams.showNext = showNext;
				coachParams.trackVisibility = trackVisibility;
				coachParams.bypassVisibilityCheck = bypassVisibilityCheck;
				coachParams.displayAsTooltip = displayAsTooltip;

				// Optional, so only set if asked for
				if (nextBtnText.length > 0)
				{
					coachParams.nextBtnText = contentStrings.nextBtnText;
				}
				cm = new Coachmark(coachParams);
			}
			catch(exc)
			{
				cm = null;

				logExc(exc);
			}

			coachmark = cm;
		}

		if (Utils.isValidProperty(coachmark))
		{
			var enabled = true;

			if (Utils.isValidProperty(params.enabledButton, Boolean))
			{
				enabled = params.enabledButton;
			}

			coachmark.setEnableNext(enabled);
			visible = coachmark.show();
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Replace escape sequences
	//
	function replaceEscapeSequences(/*String*/ inString)
	{
		var result = inString;

		// TODO: This is a start to replace escape sequences with an regex expression
		result = result.replace(/#{endl}/gi, "\n");

		return result;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get Strings
	//
	function loadContent(/*Boolean*/ inShow)
	{
		var show = inShow

		if (!contentLoaded && Utils.isValidProperty(content))
		{
			var count = 0;

			function getLocatorContent(/*String*/ inLocator, /*WorkflowContentPart*/ inPart)
			{
				var locator = inLocator;
				inPart.contentID = MonikerMapper.get().replaceSymbols(inPart.contentID); // resolves symbols in zstrings, i.e. ${TOUCHMODE}$

				try
				{
					ContentProviderManager.getContent(inPart, function(/*AString*/ inResult)
					{
						// stop processing if terminated before
						if (!terminated)
						{
							contentStrings[locator] = replaceEscapeSequences(inResult);

							if (show && --count == 0)
							{
								contentLoaded = true;
								doShow();
							}
						}
					});
				}
				catch (exc)
				{
					logExc(exc);

					if (show && --count == 0)
					{
						contentLoaded = true;
						doShow();
					}
				}
			}

			// IMPORTANT: Content strings for the coachmark can be located in any of the next locators
			//			  An according zstring entry is loaded if available in the localized table.
			var kLocators = ['title', 'text', 'debugText', 'nextBtnText'];
			var locators = [];

			Utils.forEach(kLocators, function(/*String*/ inLocator)
			{
				if (Utils.isValidProperty(content.getContent(inLocator)))
				{
					locators.push(inLocator);
				}
			});

			count = locators.length;

			Utils.forEach(locators, function(/*String*/ inLocator)
			{
				getLocatorContent(inLocator, content.getContent(inLocator));
			});

		}
		else if (inShow)
		{
			doShow();
		}
	}

}

TourCoachmarkDisplayAdapter.TYPE_ID = typeTourCoachmark;
DisplayAdapterFactory.addFactory(TourCoachmarkDisplayAdapter.TYPE_ID, function(){return new TourCoachmarkDisplayAdapter(TourCoachmarkDisplayAdapter.TYPE_ID);});

TourCoachmarkDisplayAdapter.TOOLTIP_TYPE_ID = typeTooltip;
DisplayAdapterFactory.addFactory(TourCoachmarkDisplayAdapter.TOOLTIP_TYPE_ID, function(){return new TourCoachmarkDisplayAdapter(TourCoachmarkDisplayAdapter.TOOLTIP_TYPE_ID);});
