/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

//////////////////////////////////////////////////////////////////////////////
//
// logWorkflowEvent is used to log specific state machine transitions
// into the analytics subsystem. The goal is to provide Product Management
// insight into how often the Tour is used, what the flow through the Tour
// looks like, and if there are any steps or workflows which cause a substantial
// increase in abandonment of the Tour.
//
function logWorkflowEvent(workflow, stepIndex, eventType)
{
	try
	{
		var workflowName = workflow.getID();
		var stepName = workflow.getStep(stepIndex).getName();

		if (stepName == null || stepName == "")
		{
			stepName = "Index: " + stepIndex;
		}

		ScriptedWorkflowSupport.instance.logDunamisEvent("Tour", workflowName, eventType, stepName, "", "");
	}
	catch(exc)
	{
		// just swallow anything caused by the logging code
		// print and move on
		logExc(exc);
	}
}

//////////////////////////////////////////////////////////////////////////////
//	DEBUG:	Contains most viable functions to inspect whilst running a tour
//////////////////////////////////////////////////////////////////////////////
//
// WorkflowProcessor is the actual state machine which processes a Workflow
// It's responsible, once started, to display step content in the defined
// surface and tracks user actions step by step.
//
function WorkflowProcessor(/*Workflow*/ inWorkflow, /*Function*/ inCallback)
{
	var workflow = Utils.getParamValue(inWorkflow, Utils.REQUIRED);
	var callback = Utils.getParamValue(inCallback, Utils.REQUIRED);

	var thisObj = this;

	var currentStepIndex = NaN;
	var stepIndexStack = [];
	var currentDisplays = [];
	var currentHandler = [];

	var continueCommand = null;

	//////////////////////////////////////////////////////////////////////////////
	//
	// Start workflow processing
	//
	this.start = function()
	{
		if (!isNaN(currentStepIndex))
		{
			logError('Workflow already processing, workflow id : "' + workflow.getID() + '"');
			throw new Error("Workflow already running");
		}

		// setup environment
		//
		WorkflowEnvironment.create(workflow);

		// execute workflow preActions before starting with first step
		//
		workflow.executePreActions(function()
		{
			// DEBUG:
			// Start from the beginning; index = 0 is the first step
			// You can change to a higher step, but need to ensure that the app is in the correct state
			currentStepIndex = workflow.getStepRange()[0];

			try
			{
				logInfo('Start processing, workflow id : "' + workflow.getID() + '"');
				logWorkflowEvent(workflow, currentStepIndex, "start");

				runStep();
			}
			catch(exc)
			{
				thisObj.cancel();
				logExc(exc);
			}
		});
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Restart the processing from the beginning
	//
	this.restart = function()
	{
		terminateCurrentStep();

		if (isNaN(currentStepIndex))
		{
			logInfo('Restart processing, workflow id : "' + workflow.getID() + '"');
			this.start();
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Restart the same step again
	//
	this.refresh = function()
	{
		terminateCurrentStep();

		if (!isNaN(currentStepIndex))
		{
			try
			{
				logInfo('Refresh current step, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);
				runStep();
			}
			catch(exc)
			{
				this.cancel();
				logExc(exc);
			}
		}
		else
		{
			logWarn('Attempt to refresh current step, but processing not yet started, workflow id : "' + workflow.getID() + '"');
			this.start();
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Switch to next step
	//
	this.next = function()
	{
		terminateCurrentStep();

		if (isNaN(currentStepIndex))
		{
			logWarn('Attempt to next step, but processing not yet started, workflow id : "' + workflow.getID() + '"');
			this.start();
		}
		else if ((currentStepIndex + 1) > workflow.getStepRange()[1])
		{
			logWarn('Attempt to next step, but no next step available, finish processing, workflow id : "' + workflow.getID() + '"');
			finishProcessing(WorkflowProcessor.FINISHED);
		}
		else
		{
			stepIndexStack.push(currentStepIndex);
			++currentStepIndex;

			try
			{
				logInfo('Next step, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);
				logWorkflowEvent(workflow, currentStepIndex, "display");

				runStep();
			}
			catch(exc)
			{
				this.cancel();
				logExc(exc);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Switch to previous step
	//
	this.previous = function()
	{
		terminateCurrentStep();

		if (isNaN(currentStepIndex))
		{
			logWarn('Attempt to previous step, but processing not yet started, workflow id : "' + workflow.getID() + '"');
			this.start();
		}
		else if ((currentStepIndex - 1) < 0)
		{
			logWarn('Attempt to previous step, but no previous step available, restart processing, workflow id : "' + workflow.getID() + '"');
			this.start();
		}
		else
		{
			stepIndexStack.push(currentStepIndex);
			--currentStepIndex;

			try
			{
				logInfo('Previous step, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);
				runStep();
			}
			catch(exc)
			{
				this.cancel();
				logExc(exc);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Jump to any step
	//
	this.jumpto = function(/*Number*/ inStepIndex)
	{
		terminateCurrentStep();

		if (!isNaN(inStepIndex) && inStepIndex >= 0 && inStepIndex < workflow.getStepRange()[1])
		{
			stepIndexStack.push(currentStepIndex);
			currentStepIndex = inStepIndex;

			try
			{
				logInfo('Jump to step, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);
				runStep();
			}
			catch(exc)
			{
				this.cancel();
				logExc(exc);
			}
		}
		else
		{
			logWarn('Attempt to jump to step #' + inStepIndex + ', but step is not available, workflow id : "' + workflow.getID() + '"');
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Jump back to the step we came from
	//
	this.jumpBack = function()
	{
		terminateCurrentStep();

		if (stepIndexStack.length > 0)
		{
			currentStepIndex = stepIndexStack.pop();

			try
			{
				logInfo('Jump back to step, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);
				runStep();
			}
			catch(exc)
			{
				this.cancel();
				logExc(exc);
			}
		}
		else
		{
			logWarn('Attempt to jump back, but step stack is available, workflow id : "' + workflow.getID() + '"');
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Cancel Workflow processing
	//
	this.cancel = function()
	{
		logInfo('Processing canceled, finish processing, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);
		logWorkflowEvent(workflow, currentStepIndex, "cancel");

		finishProcessing(WorkflowProcessor.CANCELED);
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Evaluate pre condition of a step
	//
	function evalStepConditions(/*WorkflowStep*/ inStep, /*Function*/ inCallback)
	{
		logInfo('Attempt to eval step pre-conditions, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);

		var callback = inCallback;

		inStep.evalConditions(function(/*String*/ inResult)
		{
			var cmd = parseCommand(inResult);
			logInfo('Step pre-conditions result command, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex + ', command: ' + cmd.cmd + ' : ' + cmd.value);
			callback(cmd);
		});
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Run processing of current step
	//
	function runStep()
	{
		if (isNaN(currentStepIndex))
		{
			throw new Error("runStep: Invalid step index: " + currentStepIndex);
		}

		WorkflowEnvironment.current.step = currentStepIndex;
		WorkflowEnvironment.current.display = [];

		// get WorkflowStep for current index, throw if not available
		//
		var step = workflow.getStep(currentStepIndex);
		Utils.throwInvalid(step, WorkflowStep);
		logInfo('Run step #' + currentStepIndex + '(' + step.getName() + ')');

		// Each step has its own label to be identifiable, so that 
		// jumps can be expressed by a label and not an absolute number as in WorkflowEnvironment.current.step
		//
		WorkflowEnvironment.stepID = step.getID();
		
		// check step conditions
		//
		evalStepConditions(step, function(result)
		{
			if (!executeCommand(result))
			{
				try
				{
					doRunStep();
				}
				catch(exc)
				{
					thisObj.cancel();
					logExc(exc);
				}
			}
		});
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	function doRunStep()
	{
		// execute pre actions
		//
		var step = workflow.getStep(currentStepIndex);
		Utils.throwInvalid(step, WorkflowStep);

		logInfo('Attempt to execute step pre-actions, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);

		step.executePreActions(function()
		{
			// display step content
			var displayTypes = step.getDisplayTypes();

			logInfo('Setup ' + displayTypes.length + ' display adapter, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);

			Utils.forEach(displayTypes, function(/*WorkflowDisplay*/ inDisplay)
			{
				try
				{
					var displayType = inDisplay.getType();

					logInfo('Create display adapter type: "' + displayType + '", workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);

					var adapter = DisplayAdapterFactory.create(displayType);

					if (Utils.isValidProperty(adapter))
					{
						logInfo('Setup step Display, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);

						currentDisplays.push(adapter);

						adapter.initialize(inDisplay.getParams());
						adapter.setContent(inDisplay.getContent());
						adapter.show();

						WorkflowEnvironment.current.display.push(adapter);
					}
					else
					{
						logError('Invalid Display Adapter "' + inDisplay.getType() + '"');
					}
				}
				catch(exc)
				{
					logExc(exc);
				}
			});

			// continuation
			//
			var stepContinuation = step.getContinuation();

			try
			{
				ContinuationManager.get().start(stepContinuation, onNextClicked);
			}
			catch(exc)
			{
				logExc(exc);
			}

			// refreshing
			ScriptedWorkflowSupport.instance.addEventListener('Coachmark.refresh', onRefreshRequested);

			// setup trigger handler
			//
			var ttypes = step.getTriggerTypes();
			var handler = [];

			logInfo('Setup ' + ttypes.length + ' trigger handler, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);

			Utils.forEach(ttypes, function(/*String*/ inType)
			{
				var trigger = step.expandTrigger(inType);

				if (Utils.isValidProperty(trigger, Array) && trigger.length > 0)
				{
					logInfo('Setup TriggerHandler, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex + ', trigger type: ' + inType);

					try
					{
						var triggerHdl = TriggerHandlerFactory.create(inType);
						triggerHdl.initialize(trigger, onTriggerHandler);
						handler.push(triggerHdl);
					}
					catch(exc)
					{
						logExc(exc);
					}
				}
			});

			if (handler.length > 0)
			{
				currentHandler = handler;
			}

			// start trigger handler
			//
			logInfo('Start all TriggerHandler, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex);

			Utils.forEach(handler, function(/*ITriggerHandler*/ inHandler)
			{
				try
				{
					inHandler.start();
				}
				catch(exc)
				{
					logExc(exc);
				}
			});

			// If there is no trigger defined for this step (handler==0) and the step also do not wait
			// for the Next button clicked (stepContinuation.type == WorkflowStepContinuation.TYPE_AUTO)
			// then move on to the next step.
			// Otherwise we end up in an undefined state and there's no way to get out of that state.
			if (handler.length == 0 &&
				(!Utils.isValidProperty(stepContinuation) ||
				(Utils.isValidProperty(stepContinuation) && stepContinuation.type == WorkflowStepContinuation.TYPE_AUTO)))
			{
				thisObj.next();
			}
		});
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	function terminateCurrentStep()
	{
		if (!isNaN(currentStepIndex))
		{
			// Terminate trigger handler
			//
			Utils.forEach(currentHandler, function(/*ITriggerHandler*/ inHandler)
			{
				try
				{
					inHandler.stop();
				}
				catch(exc)
				{
					logExc(exc);
				}
			});

			currentHandler = [];

			// Terminate displays
			//
			Utils.forEach(currentDisplays, function(/*IDisplayAdapter*/ inAdapter)
			{
				inAdapter.hide();
				inAdapter.terminate();
			});

			currentDisplays = [];
			WorkflowEnvironment.current.display = [];

			ScriptedWorkflowSupport.instance.removeEventListener('Coachmark.refresh', onRefreshRequested);

			var step = workflow.getStep(currentStepIndex);

			if (Utils.isValidProperty(step))
			{
				try
				{
					ContinuationManager.get().stop();
				}
				catch(exc)
				{
					logExc(exc);
				}
			}

			// clear command
			//
			continueCommand = null;
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Response from ITriggerHandler
	//
	function onTriggerHandler(/*TriggerHandlerStatus*/ inStatus)
	{
		var state = inStatus.state;
		var executeActions = !Utils.isValidProperty(inStatus.result);
		var currentStep    = workflow.getStep(currentStepIndex);
		var actionResult   = inStatus.result;

		// Pause trigger handler
		//
		Utils.forEach(currentHandler, function(/*ITriggerHandler*/ inHandler)
		{
			try
			{
				inHandler.pause(true);
			}
			catch(exc)
			{
				logExc(exc);
			}
		});

		function processHandlerStatus(/*String*/ inActionResult)
		{
			// parse result
			//
			var cmd = parseCommand(inActionResult);

			WorkflowEnvironment.current.event = null;

			if (state == TriggerHandlerStatus.STATUS_NEGATIVE)
			{
				// unpause trigger handler
				//
				Utils.forEach(currentHandler, function(/*ITriggerHandler*/ inHandler)
				{
					try
					{
						inHandler.pause(false);
					}
					catch(exc)
					{
						logExc(exc);
					}
				});
			}

			var continuation = currentStep.getContinuation();

			if (cmd.instant ||
				state == TriggerHandlerStatus.STATUS_NEGATIVE ||
				continuation.type == WorkflowStepContinuation.TYPE_AUTO)
			{
				// execute command immediately
				// (don't care for possibly required Next button click)
				//
				if (!executeCommand(cmd))
				{
					thisObj.next();
				}
			}
			else
			{
				continueCommand = cmd;
			}
		}

		switch (state)
		{
			case TriggerHandlerStatus.STATUS_POSITIVE:
			{
				// TODO: e.g. display step success in any way

				if (executeActions)
				{
					currentStep.executePositiveActions(processHandlerStatus);
				}
				else
				{
					processHandlerStatus(actionResult);
				}
			}
				break;

			case TriggerHandlerStatus.STATUS_NEGATIVE:
			{
				// TODO: e.g. display step failure in any way (e.g. shaking coachmark)

				if (executeActions)
				{
					currentStep.executeNegativeActions(processHandlerStatus);
				}
				else
				{
					processHandlerStatus(actionResult);
				}
			}
				break;

			default:
				processHandlerStatus(actionResult);
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// User clicked Next button in coachmark
	//
	// DEBUG: Interesting breakpoint to see what happens when user selects NEXT
	//
	function onNextClicked(/*EventObject*/ inEventObj)
	{
		logWorkflowEvent(workflow, currentStepIndex, "next_clicked");

		if (Utils.isValidProperty(continueCommand))
		{
			if (!executeCommand(continueCommand))
			{
				thisObj.next();
			}
		}
		else
		{
			thisObj.next();
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// The coachmark needs to be refreshed, likely, because it has been suppressed before
	// and needs to be shown again. See SetSuppressCoachmark in dvascriptedworkflow
	//
	function onRefreshRequested(/*EventObject*/ inEventObj)
	{
		logWorkflowEvent(workflow, currentStepIndex, "refresh_requested");
		thisObj.refresh();
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Workflow processing has finished
	//
	function finishProcessing(/*String*/ inFinishState)
	{
		var finishState = inFinishState;

		terminateCurrentStep();

		workflow.executePostActions(function()
		{
			WorkflowEnvironment.dispose();

			logInfo('Processing finished with state: "' + finishState + '"');
			callback(finishState);
		});
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Parse command from Executable result
	//
	function parseCommand(/*Any*/ inCmdSrc)
	{
		var cmd = 'continue';
		var value = null;
		var instantCmd = false;

		if (Utils.isValidProperty(inCmdSrc))
		{
			if (Utils.isType(inCmdSrc, Boolean))
			{
				cmd = (inCmdSrc ? 'continue' : 'next');
			}
			else if (Utils.isType(inCmdSrc, String))
			{
				if (inCmdSrc == '#' || inCmdSrc == '!')
				{
					cmd = 'continue';
				}
				else if (inCmdSrc.indexOf('#') == 0 || inCmdSrc.indexOf('!') == 0)
				{
					instantCmd = (inCmdSrc.indexOf('#') == 0);
					var cmdSrc = inCmdSrc.substring(1);

					switch(cmdSrc)
					{
						case 'next':
						case 'previous':
						case 'restart':
						case 'refresh':
						case 'cancel':
						case 'back':
						case 'end':
							cmd = cmdSrc;
							break;

						default:
						{
							var num = parseInt(cmdSrc);

							if (isNaN(num))
							{
								// cmdSrc is not a valid integer number, check if it is a step identifier
								num = workflow.getStepIndex(cmdSrc);
							}

							if (!isNaN(num) && num >= 0 && num < workflow.getStepRange()[1] && num != currentStepIndex)
							{
								if (num == currentStepIndex + 1)
								{
									cmd = 'next';
								}
								else if (num == currentStepIndex - 1)
								{
									cmd = 'previous';
								}
								else
								{
									cmd   = 'jump';
									value = num;
								}
							}
						}
					}
				}
			}
		}
		else
		{
			cmd = 'continue';
		}

		return {cmd : cmd, value : value, instant : instantCmd};
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Execute command, return false if command isn't executable
	//
	function executeCommand(/*Object*/ inCmd)
	{
		var executed = true;

		logInfo('Attempt to execute command, workflow id : "' + workflow.getID() + '", step : #' + currentStepIndex + ', command: ' + inCmd);

		switch (inCmd.cmd)
		{
			case 'next':
				thisObj.next();
				break;

			case 'previous':
				thisObj.previous();
				break;

			case 'restart':
				thisObj.restart();
				break;

			case 'refresh':
				thisObj.refresh();
				break;

			case 'cancel':
				thisObj.cancel();
				break;

			case 'end':
				finishProcessing(WorkflowProcessor.FINISHED);
				break;

			case 'jump':
				thisObj.jumpto(inCmd.value);
				break;

			case 'back':
				thisObj.jumpBack();
				break;

			default:
				executed = false;
		}

		return executed;
	}
}

WorkflowProcessor.FINISHED = 'finished';
WorkflowProcessor.CANCELED = 'canceled';