/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

var kGlobalScope = '_global_';

//////////////////////////////////////////////////////////////////////////////
//
// A TriggerComponent consists of one or more ITriggerDefinition (all of the
// same type) and provide it as a new ITriggerDefinition with a symbolic name
//
function TriggerComponent(/*String*/ inFormatVersion, /*Workflow*/ inWorkflow)
{
    this.name       		= '';	// symbolic name
    this.type       		= '';	// ITriggerDefinition type
    this.triggerDefinitions	= [];	// ITriggerDefinition's
	this.positiveActions 	= [];
	this.negativeActions 	= [];
	this.matchActions 		= [];

    var formatVersion = inFormatVersion;
	var workflow = inWorkflow;

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON
    //
    this.initializeJSON = function(/*String*/ inName, /*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inName);
        Utils.throwInvalid(inJSON);

        this.name = inName;

        switch (formatVersion)
        {
            case Workflow.FORMAT_VERSION_1:
                initializeJSON_V1(this, inJSON);
                break;

            default:
                logError('Invalid format version "' + formatVersion + '"');
                throw new Error("Invalid format version");
        }

        return this;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON (format version V1)
    //
    function initializeJSON_V1(/*TriggerComponent*/ inThis, /*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inThis);
        Utils.throwInvalid(inJSON);

        // trigger type (mandatory)
        //
        var type = Utils.getPropertyValue(inJSON, 'type', Utils.REQUIRED);

        // trigger definitions (mandatory)
        //
        var triggerdefs = Utils.getPropertyValue(inJSON, 'definitions', Utils.REQUIRED);
        Utils.throwInvalid(triggerdefs);

        var definitions = [];

        Utils.forEach(triggerdefs, function(definition)
        {
			var triggerDef = TriggerFactory.create(type, formatVersion, workflow);
			Utils.throwInvalid(triggerDef);

			triggerDef = triggerDef.initializeJSON(definition);

			if (Utils.isValidProperty(triggerDef))
			{
				definitions.push(triggerDef);
			}
        });

        if (definitions.length > 0)
        {
            inThis.type = type;
            inThis.triggerDefinitions = definitions;
        }


		// match actions (optional)
		//
		var matchActionsArray = Utils.getPropertyValue(inJSON, 'matchActions', Utils.OPTIONAL);

		if (Utils.isValidProperty(matchActionsArray, Array))
		{
			var matchActions = [];

			Utils.forEach(matchActionsArray, function(entry)
			{
				var action = new Executable(formatVersion, Executable.TYPE_ACTION);
				matchActions.push(action.initializeJSON(entry));
			});

			inThis.matchActions = matchActions;
		}

		// positive actions (optional)
		//
		var positiveActionsArray = Utils.getPropertyValue(inJSON, 'positiveActions', Utils.OPTIONAL);

		if (Utils.isValidProperty(positiveActionsArray, Array))
		{
			var posActions = [];

			Utils.forEach(positiveActionsArray, function(entry)
			{
				var action = new Executable(formatVersion, Executable.TYPE_ACTION);
				posActions.push(action.initializeJSON(entry));
			});

			inThis.positiveActions = posActions;
		}

		// negative actions (optional)
		//
		var negativeActionsArray = Utils.getPropertyValue(inJSON, 'negativeActions', Utils.OPTIONAL);

		if (Utils.isValidProperty(negativeActionsArray, Array))
		{
			var negActions = [];

			Utils.forEach(negativeActionsArray, function(entry)
			{
				var action = new Executable(formatVersion, Executable.TYPE_ACTION);
				negActions.push(action.initializeJSON(entry));
			});

			inThis.negativeActions = negActions;
		}
    }
}

//////////////////////////////////////////////////////////////////////////////
//
// TriggerComponentManager manage all global and (Workflow-) local TriggerComponents
// Global TriggerComponents are located in the workflows folder with the filename:
// triggercmp_{platform}_{appName}_{appVersion}.json
//
function TriggerComponentManager()
{
    this.components = {};   // [scope][trigger type][component name]

    //////////////////////////////////////////////////////////////////////////////
    //
    // Retrieve WorkflowTriggerComponent
    //
    this.get = function(/*String*/ inComponentName, /*String*/ inTriggerType, /*String*/ inWorkflowID)
    {
        var scope = Utils.getParamValue(inWorkflowID, Utils.OPTIONAL, kGlobalScope);
		var type = Utils.getParamValue(inTriggerType, Utils.OPTIONAL);

        var ret = null;

        if (Utils.isValidProperty(this.components[scope]))
        {
			if (Utils.isValidProperty(type))
			{
				if (Utils.isValidProperty(this.components[scope][inTriggerType]))
				{
					ret = this.components[scope][inTriggerType][inComponentName];
				}
			}
			else
			{
				for (var t in this.components[scope])
				{
					if (this.components[scope].hasOwnProperty(t) &&
						Utils.isValidProperty(this.components[scope][t]))
					{
						ret = this.components[scope][t][inComponentName];
						break;
					}
				}
			}
        }

        return ret;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Add WorkflowTriggerComponents
    //
    this.add = function(/*Array of WorkflowTriggerComponent*/ inComponents, /*String*/ inWorkflowID, /*Boolean*/ inOverwrite)
    {
        var components = Utils.getParamValue(inComponents, Utils.REQUIRED);
        var scope      = Utils.getParamValue(inWorkflowID, Utils.OPTIONAL, kGlobalScope);
		var overwrite  = Utils.getParamValue(inOverwrite, Utils.OPTIONAL, false);

        Utils.throwInvalid(components);

        if (!Utils.isValidProperty(this.components[scope]))
        {
            this.components[scope] = {};
        }

        var scopeCmps = this.components[scope];

        Utils.forEach(components, function(component)
        {
			logInfo('Added TriggerComponent "' + component.name + '"');
            if (!Utils.isValidProperty(scopeCmps[component.type]))
            {
                scopeCmps[component.type] = {};
            }

            if (Utils.isValidProperty(scopeCmps[component.type][component.name]) && !overwrite)
            {
                logError('Component NOT added, name exists already "' + component.name + '"');
            }
            else
            {
                scopeCmps[component.type][component.name] = component;
            }
        });
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Return the actual trigger definitions
    //
    this.getTrigger = function(/*String*/ inComponentName, /*String*/ inTriggerType, /*String*/ inWorkflowID)
    {
        var scopes = [kGlobalScope];
        var wfID = Utils.getParamValue(inWorkflowID, Utils.OPTIONAL);
        var type = Utils.getParamValue(inTriggerType, Utils.OPTIONAL, WorkflowTrigger.TYPE_SCRIPTEVENT);
        var name = Utils.getParamValue(inComponentName, Utils.REQUIRED);

        if (Utils.isValidProperty(name, String))
        {
            if (Utils.isValidProperty(wfID, String))
            {
                scopes.push(wfID);
            }

            var ret     = [];
            var thisObj = this;

            Utils.forEach(scopes, function(scope)
            {
				if (Utils.isValidProperty(thisObj.components[scope]) &&
					Utils.isValidProperty(thisObj.components[scope][type]))
				{
					var triggerCmp = thisObj.components[scope][type][name];

					if (Utils.isValidProperty(triggerCmp))
					{
						ret = ret.concat(triggerCmp.triggerDefinitions);
					}
				}
            });
        }

        return ret;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Expand all trigger of the requested component recursively
    //
    this.expandTrigger = function(/*String*/ inComponentName, /*String*/ inTriggerType, /*String*/ inWorkflowID)
    {
        var triggers = [];

		try
		{
			triggers = this.getTrigger(inComponentName, inTriggerType, inWorkflowID);
			triggers = expandAllTrigger(this, triggers);
		}
		catch(exc)
		{
            logExc(exc);
		}

        return triggers;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Recursively expand array of WorkflowTrigger
    //
    function expandAllTrigger(/*TriggerComponentManager*/ inThisObj, /*Array of WorkflowTrigger*/ inTrigger)
    {
        var ret = [];

        if (Utils.isValidProperty(inTrigger, Array))
        {
            Utils.forEach(inTrigger, function(trigger)
            {
				if (Utils.isValidProperty(trigger.triggerDefinition) &&
					trigger.triggerDefinition.isComponent())
				{
					var exp = trigger.expandTrigger();
					exp     = expandAllTrigger(inThisObj, exp);

					ret = ret.concat(exp);
				}
				else
				{
					ret.push(trigger);
				}
            });
        }

        return ret;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize
    // Load global WorkflowTriggerComponent table
    ///
    function initialize(/*TriggerComponentManager*/ inThisObj)
    {

		function registerComponents(/*Object*/ inJSON)
		{
			for (var cmp in inJSON)
			{
				if (inJSON.hasOwnProperty(cmp))
				{
					var triggerObj = inJSON[cmp];

					try
					{
						var triggerCmp = new TriggerComponent(Workflow.FORMAT_VERSION_1);
						triggerCmp.initializeJSON(cmp, triggerObj);

						if (Utils.isValidProperty(triggerCmp))
						{
							inThisObj.add([triggerCmp], kGlobalScope, true);
						}
					}
					catch (exc)
					{
						logExc(exc);
					}
				}
			}
		}

		// load common TriggerComponents
		//
		var path = ScriptedWorkflowSupport.instance.workflowRoot + '/triggercmp.json';

		try
		{
			DataRequestManager.requestJSONData(path, function(/*Boolean*/ inSuccess, /*Object*/ inJSON)
			{
				if (inSuccess && Utils.isValidProperty(inJSON))
				{
					registerComponents(inJSON);
				}

				// load platform dependent TriggerComponents
				//
				var platform = ScriptedWorkflowSupport.instance.appPlatform;

				if (platform == 'Windows' || platform == 'Macintosh')
				{
					platform = 'Desktop';
				}

				// filename pattern:
				// triggercmp_{platform}.json
				//
				var path = ScriptedWorkflowSupport.instance.workflowRoot + '/';
				var name = 'triggercmp_' + platform;
				name += '.json';
				path += name.toLowerCase();

				DataRequestManager.requestJSONData(path, function(/*Boolean*/ inSuccess, /*Object*/ inJSON)
				{
					if (inSuccess && Utils.isValidProperty(inJSON))
					{
						registerComponents(inJSON);
					}

					StartupController.get().setInitialized(StartupController.ITEM_TRIGGER_COMPONENTS);
				});
			});
		}
		catch(exc)
		{
			StartupController.get().setInitialized(StartupController.ITEM_TRIGGER_COMPONENTS);
			logExc(exc);
		}
    }

    initialize(this);
}

//////////////////////////////////////////////////////////////////////////////
//
StartupController.addItem(StartupController.ITEM_TRIGGER_COMPONENTS);
TriggerComponentManager.instance = new TriggerComponentManager();

//////////////////////////////////////////////////////////////////////////////
//
//
TriggerComponentManager.get = function()
{
    return TriggerComponentManager.instance;
}

//////////////////////////////////////////////////////////////////////////////
//
// Does component exists
//
TriggerComponentManager.exists = function(/*String*/ inComponentName, /*Workflow*/ inWorkflow, /*String*/ inTriggerType)
{
	var wfID = Utils.isValidProperty(inWorkflow) ? inWorkflow.getID() : null;

	var ret = Utils.isValidProperty(TriggerComponentManager.get().get(inComponentName, inTriggerType, wfID));

	if (!ret && Utils.isValidProperty(wfID))
	{
		ret = Utils.isValidProperty(TriggerComponentManager.get().get(inComponentName, inTriggerType));
	}

	return ret;
}

//////////////////////////////////////////////////////////////////////////////
//
// Retrieve WorkflowTriggerComponent
//
TriggerComponentManager.getComponent = function(/*String*/ inComponentName, /*String*/ inWorkflow, /*String*/ inTriggerType)
{
	var wfID = Utils.isValidProperty(inWorkflow) ? inWorkflow.getID() : null;
    var ret = TriggerComponentManager.get().get(inComponentName, inTriggerType, wfID);

	if (!Utils.isValidProperty(ret) && Utils.isValidProperty(wfID))
	{
		ret = TriggerComponentManager.get().get(inComponentName, inTriggerType);
	}

	return ret;
}

//////////////////////////////////////////////////////////////////////////////
//
// Add WorkflowTriggerComponents
//
TriggerComponentManager.add = function(/*Array of WorkflowTriggerComponent*/ inComponents, /*String*/ inWorkflowID)
{
    TriggerComponentManager.get().add(inComponents, inWorkflowID);
}

//////////////////////////////////////////////////////////////////////////////
//
// Return the actual trigger definitions
//
TriggerComponentManager.expandTrigger = function(/*String*/ inComponentName, /*String*/ inTriggerType, /*String*/ inWorkflowID)
{
    return TriggerComponentManager.get().expandTrigger(inComponentName, inTriggerType, inWorkflowID);
}