/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

//////////////////////////////////////////////////////////////////////////////
//
// A ScriptEventTrigger describes a tracked user action based on script events
// (Based on ITriggerDefinition)
//
function ScriptEventTrigger(/*String*/ inFormatVersion, /*Workflow*/ inWorkflow)
{
    this.event = '';
    this.component = '';
    this.match = '';
    this.conditions = [];

    var formatVersion = inFormatVersion;
	var workflow = inWorkflow;

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON
    //
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inJSON);

        switch (formatVersion)
        {
            case Workflow.FORMAT_VERSION_1:
                initializeJSON_V1(this, inJSON);
                break;

            default:
                logError('Invalid format version "' + formatVersion + '"');
                throw new Error("Invalid format version");
        }

        return this;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Does the trigger point to a component?
    //
    this.isComponent = function()
    {
        return (this.component.length > 0);
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Clone object
    //
    this.clone = function()
    {
        var ret = new ScriptEventTrigger(formatVersion, workflow);
        ret.event       = this.event;
        ret.component   = this.component;
        ret.match       = this.match;
        ret.conditions  = this.conditions;

        return ret;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Return a copy of this object or, if this trigger points to a component
    // then return all trgger of the component.
    //
    this.expandTrigger = function(/*String*/ inWorkflowID)
    {
        var ret = [];

        if (this.isComponent())
        {
            // retrieve copies of all related (and expanded) trigger objects
            //
            var triggers = TriggerComponentManager.expandTrigger(this.component, ScriptEventTrigger.TYPE_SCRIPTEVENT, inWorkflowID);

            var triggerMatch = this.match;

            Utils.forEach(triggers, function(trigger)
            {
				trigger = trigger.clone();

                if (triggerMatch == ScriptEventTrigger.MATCH_IGNORE)
                {
                    // if the component is used with the ignore flag then
                    // set the ignore flag of each trigger of the component
                    //
                    trigger.match = ScriptEventTrigger.MATCH_IGNORE;
                }
                else if (triggerMatch != ScriptEventTrigger.MATCH_POSITIVE &&
                    trigger.match != ScriptEventTrigger.MATCH_IGNORE)
                {
                    // if the component is used with a negative match
                    // then turn the match of each trigger of the component around
                    //
                    trigger.match = (trigger.match == ScriptEventTrigger.MATCH_POSITIVE ? ScriptEventTrigger.MATCH_NEGATIVE : ScriptEventTrigger.MATCH_POSITIVE);
                }

                ret.push(trigger);
            });
        }
        else
        {
            ret.push(this.clone());
        }

        return ret;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    //
    //
    this.validate = function(/*EventObject*/ inEventObj, /*Function*/ inCallback)
    {
        if (inEventObj.type == this.event)
        {
            if (this.conditions.length > 0)
            {
				logInfo('Hit incomming script event (w/ conditions), type: "' + inEventObj.type + '"');

                var callback = inCallback;
                this.evalConditions(function(result)
                {
                    callback(result);
                });
            }
            else
            {
				logInfo('Hit incomming script event (w/o conditions), type: "' + inEventObj.type + '"');
				inCallback(true);
            }
        }
        else
        {
			logInfo('NO hit incomming script event, type: "' + inEventObj.type + '"');
            inCallback(false);
        }
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Evaluate all conditions
    // Returns 'false' if any condition returns 'false'
    //
    this.evalConditions = function(/*Function*/ inCallback)
    {
        var count = this.conditions.length;

        if (count > 0)
        {
            // execute each condition executable in the order they were defined
            //
            Utils.forEach(this.conditions, function(/*Executable*/ inCondition)
            {
                var results = [];

                inCondition.execute(function(ret)
                {
                    // store each result which is not 'undefined'
                    //
                    if (Utils.isValidProperty(ret) && ret.success && Utils.isValidProperty(ret.result) && ret.result != true)
                    {
                        results.push(ret.result == true);
                    }

                    // if all conditions were evaluated then call back the result
                    //
                    if (--count <= 0 && Utils.isValidProperty(inCallback))
                    {
                        inCallback(results.length > 0 ? results[0] : true);
                    }
                });
            });
        }
        else if (Utils.isValidProperty(inCallback))
        {
            // no conditions available, call back immediately
            //
            inCallback(true);
        }
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Initialize with JSON (format version V1)
    //
    function initializeJSON_V1(/*ScriptEventTrigger*/ inThis, /*[Object]*/ inJSON)
    {
        Utils.throwInvalid(inThis);
        Utils.throwInvalid(inJSON);

        // event type (mandatory)
        // Should be either a script event name or a trigger
        // ScriptEventTriggerComponent name
        //
        var event = Utils.getPropertyValue(inJSON, 'event', Utils.REQUIRED);

		if (!ScriptEventTypes.get().exists(event) &&
			!TriggerComponentManager.exists(event, workflow, ScriptEventTrigger.TYPE_SCRIPTEVENT))
        {
            logError('Unknown script event type "' + event + '"');
            throw new Error("Invalid event type");
        }

        if (ScriptEventTypes.get().exists(event))
        {
            inThis.event = event;
        }
        else if (TriggerComponentManager.exists(event, workflow, ScriptEventTrigger.TYPE_SCRIPTEVENT))
        {
            inThis.component = event;
        }

        // event match (optional)
        //
        var kSupportedMatch = [ScriptEventTrigger.MATCH_POSITIVE,
			ScriptEventTrigger.MATCH_NEGATIVE,
			ScriptEventTrigger.MATCH_IGNORE];
        var match = Utils.getPropertyValue(inJSON, 'match', Utils.OPTIONAL, ScriptEventTrigger.MATCH_POSITIVE);

        if (kSupportedMatch.indexOf(match) < 0)
        {
            logError('Invalid trigger match "' + match + '"');
            throw new Error("Invalid trigger match");
        }

        inThis.match = match;

        // trigger conditions (optional)
        //
        var conditionsArray = Utils.getPropertyValue(inJSON, 'conditions', Utils.OPTIONAL);

        if (Utils.isValidProperty(conditionsArray, Array))
        {
            var conditions = [];

            Utils.forEach(conditionsArray, function(entry)
            {
                var condition = new Executable(formatVersion, Executable.TYPE_CONDITION);
                conditions.push(condition.initializeJSON(entry));
            });

            inThis.conditions = conditions;
        }
    }
}

ScriptEventTrigger.MATCH_POSITIVE   = 'positive';
ScriptEventTrigger.MATCH_NEGATIVE   = 'negative';
ScriptEventTrigger.MATCH_IGNORE     = 'ignore';

// Register trigger
//
ScriptEventTrigger.TYPE_SCRIPTEVENT = 'scriptevent';

TriggerFactory.addFactory(ScriptEventTrigger.TYPE_SCRIPTEVENT, function(/*String*/ inFormatVersion, /*Workflow*/ inWorkflow){return new ScriptEventTrigger(inFormatVersion, inWorkflow);});
