/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

//////////////////////////////////////////////////////////////////////////////
//
// The MonikerMapper provides a mapping between a moniker string and any
// other value.
// The mapping is based on a mapping file of the format:
//	{
//		"moniker_1" : value_1,
//		"moniker_2" : value_2,
//		:
//  }
//
// The moniker mapping file should be located in the Workflows root folder
// with the filename: moniker_{platform}_{appName}_{appVersion}.json
//
function MonikerMapper()
{
	var mtable = {};		// Array of { moniker1 : value1 , moniker2 : value2, ... }
	var initialized = false;

	//////////////////////////////////////////////////////////////////////////////
	//
	// Is inValue a moniker?
	//
	this.isMoniker = function(/*String*/ inValue)
	{
		return mtable.hasOwnProperty(inValue);
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Is there a moniker for inValue?
	//
	this.hasMoniker = function(/*String*/ inValue)
	{
		return (findMatches(inValue).length > 0);
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Return Array of moniker for inValue
	// (return inValue if it is already a moniker)
	//
	this.getMoniker = function(/*String*/ inValue)
	{
		var ret = [inValue];

		if (this.hasMoniker(inValue))
		{
			ret = findMatches(inValue);
		}

		return ret;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Return value for inMoniker
	// (return inMoniker if it is already the value)
	//
	this.getValue = function(/*String*/ inMoniker)
	{
		var ret = inMoniker;
		inMoniker = this.replaceSymbols(inMoniker);
		if (this.isMoniker(inMoniker))
		{
			ret = mtable[inMoniker];
			ret = this.replaceSymbols(ret);
		}

		return ret;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	function isTouchMode()
	{
		// Some of the strings differ by expressing the touch centric user interaction.
		// This currently is bound to platforms, i.e. desktop uses mouse input, but is rather runtime dependent.
		// For now, we don't query into ScriptedWorkflowSupport.instance, but might need to do in the future.
		var platform = ScriptedWorkflowSupport.instance.appPlatform;
		if (platform == 'Windows' || platform == 'Macintosh')
		{
			return false;
		}

		return true;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Replace all known symbols withion the string
	//
	this.replaceSymbols = function(/*String*/ inValue)
	{
		var kSymbols =[ { symbol : '${VIEWMODE}$',
							replacement : function()
											{
												return (ScriptedWorkflowSupport.instance.compact ? 'compact' : 'normal');
											} },
						{ symbol : '${TOUCHMODE}$',
						replacement : function()
										{
											return (isTouchMode() ? 'touch' : 'click');
										} }
						];

		// escape for RegExp
		//
		function escapeX(/*String*/ inStr)
		{
			var str = inStr;

			for (var p=0; p<str.length; p++)
			{
				switch(str.charAt(p))
				{
					case '/':
					case '\\':
					case '[':
					case ']':
					case '{':
					case '}':
					case '!':
					case '$':
					case '%':
					case '&':
					case '?':
					{
						if (p >= 0 && str.charAt(p-1) != '\\')
						{
							var tmp = str.substr(0, p);
							tmp += '\\';
							tmp += str.substr(p);
							str = tmp;
						}
					}
						break;
				}
			}

			return str;
		}

		var value = inValue;

		Utils.forEach (kSymbols, function(inSymbol)
		{
			var regex = new RegExp(escapeX(inSymbol.symbol), 'g');
			value = value.replace(regex, inSymbol.replacement());
		});

		return value;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Find all moniker which match to the input value
	// (since different moniker to have the same value the return value is an Array)
	//
	function findMatches(/*String*/ inValue)
	{
		var ret = [];

		for (var moniker in mtable)
		{
			if (mtable.hasOwnProperty(moniker) &&
				mtable[moniker].length > 0 &&
				inValue.length >= mtable[moniker].length &&
				inValue.substr(inValue.length - mtable[moniker].length) == mtable[moniker])
			{
				ret.push(moniker);
			}
		}

		return ret;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	function loadMap()
	{
		// filename pattern:
		// moniker.json
		//
		var path = ScriptedWorkflowSupport.instance.workflowRoot + '/moniker.json';

		logInfo('About to load common moniker mapping file from ' + path);

		try
		{
			DataRequestManager.requestJSONData(path, function(/*Boolean*/ inSuccess, /*Object*/ inJSON)
			{
				initialized = true;

				logInfo('Loading common moniker mapping file ' + (inSuccess ? 'succeeded' : 'failed') + ', ' + path);

				if (inSuccess && Utils.isValidProperty(inJSON))
				{
					mtable = inJSON;
				}

				// load platform dependent file
				//
				var platform = ScriptedWorkflowSupport.instance.appPlatform;

				if (platform == 'Windows' || platform == 'Macintosh')
				{
					platform = 'Desktop';
				}

				// filename pattern:
				// moniker_{platform}.json
				//
				var path = ScriptedWorkflowSupport.instance.workflowRoot + '/';
				var name = 'moniker_' + platform;
				name += '.json';
				path += name.toLowerCase();

				logInfo('About to load moniker mapping file from ' + path);

				DataRequestManager.requestJSONData(path, function(/*Boolean*/ inSuccess, /*Object*/ inJSON)
				{
					logInfo('Loading moniker mapping file ' + (inSuccess ? 'succeeded' : 'failed') + ', ' + path);

					if (inSuccess && Utils.isValidProperty(inJSON))
					{
						for (var name in inJSON)
						{
							if (inJSON.hasOwnProperty(name))
							{
								mtable[name] = inJSON[name];
							}
						}
					}

					StartupController.get().setInitialized(StartupController.ITEM_MONIKER);
				});
			});
		}
		catch(exc)
		{
			StartupController.get().setInitialized(StartupController.ITEM_MONIKER);
			logExc(exc);
		}
	}

	loadMap();
}

//////////////////////////////////////////////////////////////////////////////
//
StartupController.addItem(StartupController.ITEM_MONIKER);
MonikerMapper.instance = new MonikerMapper();

//////////////////////////////////////////////////////////////////////////////
//
MonikerMapper.get = function()
{
	return MonikerMapper.instance;
}