/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

//////////////////////////////////////////////////////////////////////////////
//
// The ExecutorManager deals with any type of action defined in the tour.
//
// Typical element of a guided tour: 
//
//		"preActions" :
//		[
//			{
//				"type" : "script"
//				"eval" : "ScriptedWorkflowSupport.instance.setOverrideData('undo_button', 0);"
//			}
//		]
//
//	"preActions" would be one type of actions next to conditions.
//
//	For example, the preActions are executed prior of a step processing.

//////////////////////////////////////////////////////////////////////////////
//
//
function ExecutionResult(/*Any*/ inResult, /*Boolean*/ inSuccess, /*String*/ inMessage)
{
	this.success    = Utils.getParamValue(inSuccess, Utils.OPTIONAL, (Utils.isValidProperty(inResult) ? true : false));
	this.result     = Utils.getParamValue(inResult, Utils.OPTIONAL);
	this.message    = Utils.getParamValue(inMessage, Utils.OPTIONAL, '');
}

function ExecutorManager()
{
	var executors = {};

	//////////////////////////////////////////////////////////////////////////////
	//
	// Execute
	//
	this.execute = function(/*String*/ inType, /*String*/ inEvalSrc, /*Boolean*/ inCondition, /*Function*/ inCallback)
	{
		if (Utils.isValidProperty(executors[inType]))
		{
			executors[inType].execute(inEvalSrc, inCondition, inCallback);
		}
		else
		{
			logError('Unknown executor type: ' + inType);

			var res = new ExecutionResult(null, false, 'Unknown executor type: ' + inType);

			if (Utils.isValidProperty(inCallback))
			{
				inCallback(res);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Executor type exists?
	//
	this.exists = function(/*String*/ inType)
	{
		return Utils.isValidProperty(executors[inType]);
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Add executor
	//
	this.add = function(/*String*/ inType, /*IExecutor*/ inExecutor)
	{
		var type = Utils.getParamValue(inType, Utils.REQUIRED);
		var executor = Utils.getParamValue(inExecutor, Utils.REQUIRED);

		if (Utils.isValidProperty(executors[type]))
		{
			logWarn('Executor already available for type: ' + type);
		}

		executors[type] = executor;
	}
}

//////////////////////////////////////////////////////////////////////////////
//
//
ExecutorManager.instance = null;
ExecutorManager.get = function()
{
	if (!Utils.isValidProperty(ExecutorManager.instance))
	{
		ExecutorManager.instance = new ExecutorManager();
	}

	return ExecutorManager.instance;
}

//////////////////////////////////////////////////////////////////////////////
//
// Execute
//
ExecutorManager.execute = function(/*String*/ inType, /*String*/ inEvalSrc, /*Boolean*/ inCondition, /*Function*/ inCallback)
{
	ExecutorManager.get().execute(inType, inEvalSrc, inCondition, inCallback);
}

//////////////////////////////////////////////////////////////////////////////
//
// Add executor
//
ExecutorManager.add = function(/*String*/ inType, /*IExecutor*/ inExecutor)
{
	ExecutorManager.get().add(inType, inExecutor);
}

//////////////////////////////////////////////////////////////////////////////
//
// Executor type exists?
//
ExecutorManager.exists = function(/*String*/ inType)
{
	return ExecutorManager.get().exists(inType);
}