/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

//////////////////////////////////////////////////////////////////////////////
//
// The DataRequestManager retrieve file data from any location
// (local or remote)
//
function DataRequestManager()
{
	var requests = {};	// requests[bucketID] = {reqIDs : [], paths : [], callback : callback};
	var nextID = 0;

	ScriptedWorkflowSupport.instance.addEventListener('ScriptedWorkflowSupport.requestResult', onDataLoaded);
	ScriptedWorkflowSupport.instance.addEventListener('ScriptedWorkflowSupport.requestFailed', onLoadFailure);

	//////////////////////////////////////////////////////////////////////////////
	//
	// Retrieve file data from inPath
	//
	this.requestData = function(/*String*/ inPath, /*Function*/ inCallback)
	{
		this.requestDataAlt([inPath], inCallback);
	}

	//////////////////////////////////////////////////////////////////////////////
	// Public:
	// Retrieve file data from any locations of inPaths
	// Starting with the first path trying as long until data was retrieved
	// successfully or all pathes were tried.
	//
	// DEBUG: Mostlikely function used to read data asynchronously.
	//		  See for example loadTable() in GuidedtourContentProvider.js
	//
	this.requestDataAlt = function(/*Array of String*/ inPaths, /*Function*/ inCallback)
	{
		var paths = Utils.getParamValue(inPaths, Utils.REQUIRED);
		var callback = Utils.getParamValue(inCallback, Utils.REQUIRED);

		Utils.throwInvalid(paths, Array);
		Utils.throwInvalid(callback, Function);

		var bucketID = generateID();
		requests[bucketID] = {reqIDs : [], paths : [], callback : callback};

		logInfo('Request bucket of ' + paths.length + ' requests, bucked id #' + bucketID);

		Utils.forEach(paths, function(/*String*/ inPath)
		{
			var reqID = generateID();
			requests[bucketID].reqIDs.push(reqID);
			requests[bucketID].paths.push(inPath);
		});

		doRequestData(requests[bucketID].reqIDs[0], requests[bucketID].paths[0]);
	}

	//////////////////////////////////////////////////////////////////////////////
	// Public:
	// Retrieve JSON data from file at inPath
	//
	this.requestJSONData = function(/*String*/ inPath, /*Function*/ inCallback)
	{
		var callback = Utils.getParamValue(inCallback, Utils.REQUIRED);

		this.requestData(inPath, function(/*Boolean*/ inSuccess, /*String*/ inData)
		{
			var json = null;

			if (inSuccess && Utils.isValidProperty(inData))
			{
				try
				{
					json = JSON.parse(inData);
				}
				catch (exc)
				{
					logExc(exc);
				}
			}

			callback(inSuccess, json);
		});
	}

	//////////////////////////////////////////////////////////////////////////////
	// Private:
	// Send request
	//
	function doRequestData(/*Number*/ inID, /*String*/ inPath)
	{
		logInfo('Request data (id #' + inID + ') from ' + inPath);
		ScriptedWorkflowSupport.instance.requestData(inID, inPath);
	}

	//////////////////////////////////////////////////////////////////////////////
	// Private:
	// File data loaded successfully
	//
	function onDataLoaded(/*EventObject*/ inEventObj)
	{
		logInfo('Request data succeeded, id #' + inEventObj.id);
		var bucketID = findBucket(inEventObj.id);

		if (bucketID >= 0)
		{
			requests[bucketID].callback(true, inEventObj.data);
			delete requests[bucketID];
		}
		else
		{
			logWarn('No valid buckedID for requestID #' + inEventObj.id);
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	// Private:
	// Failed loading file data
	//
	function onLoadFailure(/*EventObject*/ inEventObj)
	{
		logInfo('Request data failed, id #' + inEventObj.id);
		var bucketID = findBucket(inEventObj.id);

		if (bucketID >= 0)
		{
			var bucket = requests[bucketID];

			if (bucket.reqIDs.length <= 1)
			{
				logWarn('Invalid bucket number: #' + bucketID)

				bucket.callback(false, inEventObj.data);
				delete requests[bucketID];
			}
			else
			{
				var index = bucket.reqIDs.indexOf(inEventObj.id);

				if (index >= 0)
				{
					bucket.reqIDs.splice(index, 1);
					bucket.paths.splice(index, 1);
				}

				doRequestData(bucket.reqIDs[0], bucket.paths[0]);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	// Private:
	//
	function findBucket(/*Number*/ inRequestID)
	{
		for (var bucketID in requests)
		{
			if (requests.hasOwnProperty(bucketID))
			{
				if (requests[bucketID].reqIDs.indexOf(inRequestID) >= 0)
				{
					return bucketID;
				}
			}
		}

		return -1;
	}

	//////////////////////////////////////////////////////////////////////////////
	// Private:
	//
	function generateID()
	{
		var id = ++nextID;

		if (nextID == Number.MAX_VALUE)
		{
			nextID = 0;
			id = generateID();
		}

		return id;
	}
}

//////////////////////////////////////////////////////////////////////////////
// Public:
//
//
DataRequestManager.instance = null;
DataRequestManager.get = function()
{
	if (!Utils.isValidProperty(DataRequestManager.instance))
	{
		DataRequestManager.instance = new DataRequestManager();
	}

	return DataRequestManager.instance;
}

//////////////////////////////////////////////////////////////////////////////
// Public:
// Retrieve file data from inPath
//
DataRequestManager.requestData = function(/*String*/ inPath, /*Function*/ inCallback)
{
	DataRequestManager.get().requestData(inPath, inCallback);
}

//////////////////////////////////////////////////////////////////////////////
// Public:
// Retrieve file data from any locations of inPaths
//
DataRequestManager.requestDataAlt = function(/*Array of String*/ inPaths, /*Function*/ inCallback)
{
	DataRequestManager.get().requestDataAlt(inPaths, inCallback);
}

//////////////////////////////////////////////////////////////////////////////
// Public:
// Retrieve JSON data from file at inPath
//
DataRequestManager.requestJSONData = function(/*Array of String*/ inPaths, /*Function*/ inCallback)
{
	DataRequestManager.get().requestJSONData(inPaths, inCallback);
}