/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

// The purpose of the DisplayAdapterFactory is to delegate display tasks
// not only to a coachmark, but potentially also to other UI views.

function DisplayAdapterFactory()
{
    var factories = {};

    //////////////////////////////////////////////////////////////////////////////
    //
    // Create adapter for display type
    //
    this.create = function(/*String*/ inDisplayType)
    {
        var type = Utils.getParamValue(inDisplayType, Utils.REQUIRED);

        Utils.throwInvalid(factories[type]);

        return factories[type]();
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Return Array of available display adapter type names
    //
    this.getTypes = function()
    {
        var types = [];

        for (var type in factories)
        {
            types.push(type);
        }

        return types;
    }

    //////////////////////////////////////////////////////////////////////////////
    //
    // Add factory function
    //
    this.addFactory = function(/*String*/ inDisplayType, /*Function*/ inFactory)
    {
		logInfo('Register factory, type: "' + inDisplayType + '"');

        var type    = Utils.getParamValue(inDisplayType, Utils.REQUIRED);
        var factory = Utils.getParamValue(inFactory, Utils.REQUIRED);

        Utils.throwInvalid(type, String);
        Utils.throwInvalid(factory, Function);

        if (!Utils.isValidProperty(factories[type]))
        {
            factories[type] = factory;
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
//
//
DisplayAdapterFactory.instance = null;
DisplayAdapterFactory.get = function()
{
	if (!Utils.isValidProperty(DisplayAdapterFactory.instance))
	{
		DisplayAdapterFactory.instance = new DisplayAdapterFactory();
	}

	return DisplayAdapterFactory.instance;
}

//////////////////////////////////////////////////////////////////////////////
//
// Retrieve WorkflowDisplayAdapter for display type
//
DisplayAdapterFactory.create = function(/*String*/ inDisplayType)
{
    return DisplayAdapterFactory.get().create(inDisplayType);
}

//////////////////////////////////////////////////////////////////////////////
//
// Retrieve available type names
//
DisplayAdapterFactory.getTypes = function()
{
	return DisplayAdapterFactory.get().getTypes();
}

//////////////////////////////////////////////////////////////////////////////
//
// Register IDisplayAdapter factory function
//
//  For example, see CoachmarkDisplayAdapter.js
//
DisplayAdapterFactory.addFactory = function(/*String*/ inDisplayType, /*Function*/ inFactory)
{
	try
	{
		DisplayAdapterFactory.get().addFactory(inDisplayType, inFactory);
	}
	catch(exc)
	{
		logExc(exc);
	}
}
