/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2018 Adobe                                               */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE: All information contained herein is, and remains            */
/* the property of Adobe and its suppliers, if any. The intellectual   */
/* and technical concepts contained herein are proprietary to Adobe    */
/* and its suppliers and are protected by all applicable intellectual  */
/* property laws, including trade secret and copyright laws.           */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe.                                                         */
/*                                                                     */
/***********************************************************************/

var Utils = {};

Utils.runtime = 'v8';    // in case of Extendscript the value will be 'es' otherwise 'v8'

Utils.OPTIONAL = 0;
Utils.REQUIRED = 1;

//////////////////////////////////////////////////////////////////////////////
//
// Return false if parameter prop is invalid,
// i.e. prop is not defined or prop == null
//
Utils.isValidProperty = function(/*Any*/ inProperty, /*Class*/ inType)
{
    // Verify if existing to prevent exception if in global space
    if (typeof(inProperty) === 'undefined')
    {
        return false;
    }

    // Verify if null, but it means existing
    if (inProperty === null)
    {
        return false;
    }

    // Verify type
    if (Utils.isValidProperty(inType))
    {
        return Utils.isType(inProperty, inType);
    }

    return true;
}

//////////////////////////////////////////////////////////////////////////////
//
// Throw exception if parameter inObj is invalid
//
Utils.throwInvalid = function(/*Any*/ inObj, /*Class*/ inType)
{
    if (!Utils.isValidProperty(inObj))
    {
        throw new Error("Invalid Object");
    }

    if (Utils.isValidProperty(inType) && !Utils.isType(inObj, inType))
    {
        throw new Error("Invalid Object Type");
    }
}

//////////////////////////////////////////////////////////////////////////////
//
// Return inParam value of its default value
//
// If inRequired is Utils.REQUIRED then throw if there's no value
//
Utils.getParamValue = function(/*Any*/ inParam, /*Number [Utils.OPTIONAL]*/ inRequired, /*Any*/ inDefault)
{
    var ret = inParam;

	if (!Utils.isValidProperty(ret))
	{
		var required = Utils.isValidProperty(inRequired) ? (inRequired == Utils.REQUIRED) : false;

		if (required)
		{
			Utils.throwInvalid(ret);
		}
		else
		{
			ret = inDefault;
		}
	}

    return ret;
}

//////////////////////////////////////////////////////////////////////////////
//
// Return value of object property.
//
// If inRequired is Utils.REQUIRED then throw if there's no value
//
Utils.getPropertyValue = function(/*Object*/ inObject, /*String*/ inPropertyName, /*Number [Utils.OPTIONAL]*/ inRequired, /*Any*/ inDefaultValue)
{
    inObject        = Utils.getParamValue(inObject, Utils.REQUIRED);
    inPropertyName  = Utils.getParamValue(inPropertyName, Utils.REQUIRED);
	inRequired      = Utils.getParamValue(inRequired, Utils.OPTIONAL, Utils.OPTIONAL);
    inDefaultValue  = Utils.getParamValue(inDefaultValue, Utils.OPTIONAL);

    var ret;

    if (Utils.isValidProperty(inObject[inPropertyName]))
    {
        // Property is valid, return its value
        //
        ret = inObject[inPropertyName];
    }
    else
    {
        // Property is invalid, use the default value
        //
        ret = inDefaultValue;

        if (inRequired == Utils.REQUIRED)
        {
            // Property is mandatory, throw if return value is invalid
            //
            Utils.throwInvalid(ret);
        }
    }

    return ret;
}

//////////////////////////////////////////////////////////////////////////////
//
// Return true if inValue is of type inType
//
Utils.isType = function(/*Any*/ inValue, /*Class*/ inType)
{
    var ret = Utils.isValidProperty(inValue) && Utils.isValidProperty(inType);

    if (ret)
    {
        if (inType == String)
        {
            // String type requires special handling
            if (Utils.isValidProperty(inValue.constructor))
            {
                ret = (inValue.constructor.name == 'String');
            }
        }
        else if (inType == Boolean)
        {
            ret = (typeof inValue == 'boolean');
        }
        else
        {
            ret = inType.prototype.isPrototypeOf(inValue);
        }
    }

    return ret;
}

//////////////////////////////////////////////////////////////////////////////
//
// Calls callback function inProcessFct for each entry of Array inArray
//
// DEBUG: Important, since it is function step over doesn't step inside
//
Utils.forEach = function(inArray, inProcessFct)
{
    Utils.throwInvalid(inArray, Array);
    Utils.throwInvalid(inProcessFct, Function);

    for (var i=0; i<inArray.length; i++)
    {
        inProcessFct(inArray[i]);
    }
}

//////////////////////////////////////////////////////////////////////////////
//
// Clone a vanilla Object
// ! no deep cloning !
//
// Objects are transfered by reference. To prevent accidental change, use cloneObject() to create a new copy
//
Utils.cloneObject = function(/*Object*/ inObject)
{
	var ret = inObject;

	if (Utils.isValidProperty(inObject, Object))
	{
		ret = {};

		for (var p in inObject)
		{
			if (inObject.hasOwnProperty(p))
			{
				ret[p] = inObject[p];
			}
		}
	}

	return ret;
}

//////////////////////////////////////////////////////////////////////////////
//
// Evaluate runtime
//
Utils.runtime = (typeof($) !== 'undefined' && typeof($.engineName) !== 'undefined' ? 'es' : 'v8');
