define(function (require) {

    // Brackets modules
    var FileSystem      = brackets.getModule("filesystem/FileSystem"),
        FileUtils       = brackets.getModule("file/FileUtils"),
        ProjectManager  = brackets.getModule("project/ProjectManager");

    // Local modules
    var Promise         = require("bluebird"),
        ErrorHandler    = require("src/ErrorHandler"),
        Events          = require("src/Events"),
        EventEmitter    = require("src/EventEmitter"),
        ExpectedError   = require("src/ExpectedError"),
        ProgressDialog  = require("src/dialogs/Progress"),
        CloneDialog     = require("src/dialogs/Clone"),
        Git             = require("src/git/Git"),
        Preferences     = require("src/Preferences"),
        DWGit           = require("src/DWGit/DWGit"),
        DWGitEvents     = require("src/DWGit/DWGitEvents"),
        Utils           = require("src/Utils"),
        Strings         = require("strings"),
        URI             = require("URI");

    // Templates
    var gitignoreTemplate = require("text!templates/default-gitignore");

    // Module variables

    // Implementation

    function createGitIgnore(gitRoot) {
        var gitIgnorePath = gitRoot + ".gitignore";
        Utils.pathExists(gitIgnorePath).then(function (exists) {
            if (!exists) {
                dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_CREATE_GITIGNORE, gitRoot);
            }
        });
    }

    function stageGitIgnore(msg) {
        return createGitIgnore().then(function () {
            return Git.stage(".gitignore");
        });
    }

    function handleGitInit(projectPath) {
        Utils.isProjectRootWritable(projectPath).then(function (writable) {
            if (!writable) {
                throw new ExpectedError("Folder " + (projectPath || Utils.getProjectRoot()) + " is not writable!");
            }
            return Git.init(projectPath).catch(function (err) {
            // below code can be used later.
//                if (ErrorHandler.contains(err, "Please tell me who you are")) {
//                    var defer = Promise.defer();
//                    EventEmitter.emit(Events.GIT_CHANGE_USERNAME, null, function () {
//                        EventEmitter.emit(Events.GIT_CHANGE_EMAIL, null, function () {
//                            Git.init(projectPath).then(function (result) {
//                                defer.resolve(result);
//                            }).catch(function (err) {
//                                defer.reject(err);
//                            });
//                        });
//                    });
//                    return defer.promise;
//                }

                throw err;

            });
        }).then(function () {
            DWGit.onCreateGitRepo(projectPath);
            createGitIgnore(projectPath);
        }).catch(function (err) {
            ErrorHandler.showError(err, Strings.ERROR_INIT_REPO);
        });
    }

    // This checks if the project root is empty (to let Git clone repositories)
    function isProjectRootEmpty() {
        return new Promise(function (resolve, reject) {
            ProjectManager.getProjectRoot().getContents(function (err, entries) {
                if (err) {
                    return reject(err);
                }
                resolve(entries.length === 0);
            });
        });
    }

    /*function handleGitClone() {
        var $gitPanel = $("#git-panel");
        var $cloneButton = $gitPanel.find(".git-clone");
        $cloneButton.prop("disabled", true);
        isProjectRootEmpty().then(function (isEmpty) {
            if (isEmpty) {
                CloneDialog.show().then(function (cloneConfig) {
                    var q = Promise.resolve();
                    // put username and password into remote url
                    var remoteUrl = cloneConfig.remoteUrl;
                    if (cloneConfig.remoteUrlNew) {
                        remoteUrl = cloneConfig.remoteUrlNew;
                    }

                    // do the clone
                    q = q.then(function () {
                        return ProgressDialog.show(Git.clone(remoteUrl, "."));
                    }).catch(function (err) {
                        ErrorHandler.showError(err, "Cloning remote repository failed!");
                    });

                    // restore original url if desired
                    if (cloneConfig.remoteUrlRestore) {
                        q = q.then(function () {
                            return Git.setRemoteUrl(cloneConfig.remote, cloneConfig.remoteUrlRestore);
                        });
                    }

                    return q.finally(function () {
                        EventEmitter.emit(Events.REFRESH_ALL);
                    });
                }).catch(function (err) {
                    // when dialog is cancelled, there's no error
                    if (err) { ErrorHandler.showError(err, "Cloning remote repository failed!"); }
                });

            } else {
                var err = new ExpectedError("Project root is not empty, be sure you have deleted hidden files");
                ErrorHandler.showError(err, "Cloning remote repository failed!");
            }
        }).catch(function (err) {
            ErrorHandler.showError(err);
        }).finally(function () {
            $cloneButton.prop("disabled", false);
        });
    }*/

    function handleGitClone(siteRoot, remoteRepoUrl, username, password, bRemember) {
        var q = Promise.resolve();

        var remoteUrl = remoteRepoUrl;
        var remoteUrlNew;
        if (remoteRepoUrl.match(/^https?:/)) {
            var uri = new URI(remoteRepoUrl);
            uri.username(username);
            uri.password(password);
            remoteUrlNew = uri.toString();
            // assign remoteUrlNew only if it's different from the original url
            if (remoteUrlNew && remoteUrl !== remoteUrlNew) {
                remoteUrl = remoteUrlNew;
            }
        }

        // do the clone
        var cloneFailed = false;
        q = q.then(function () {
            return ProgressDialog.show(Git.clone(remoteUrl, siteRoot));
        }).catch(function (err) {
            var title = Strings.ERROR_CLONE_REPO;
            var errorBody;
            if (typeof err === "string") {
                errorBody = err;
            } else if (err instanceof Error) {
                errorBody = Utils.encodeSensitiveInformation(err.toString());
            }
            dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_CLONE_ERROR, (title + "\n" + errorBody));
            cloneFailed = true;
        });

        // restore original url if it was http clone
        if (remoteUrlNew) {
            q = q.then(function () {
                return Git.setRemoteUrl("origin", remoteRepoUrl);
            });
        }

        return q.finally(function () {
            if(cloneFailed === false) {
                if(bRemember) {
                    DWGit.onCreateGitRepo(siteRoot, username, password);
                } else {
                    DWGit.onCreateGitRepo(siteRoot);
                }
            }
        });
    }

    // Event subscriptions
    EventEmitter.on(Events.HANDLE_GIT_INIT, function (projectPath) {
        handleGitInit(projectPath);
    });
    EventEmitter.on(Events.HANDLE_GIT_CLONE, function (siteRoot, remoteRepoUrl, username, password, bRemember) {
        handleGitClone(siteRoot, remoteRepoUrl, username, password, bRemember);
    });
    EventEmitter.on(Events.GIT_NO_BRANCH_EXISTS, function () {
        stageGitIgnore();
    });

});
