
define(function (require, exports) {
    // Brackets modules
    var _               = brackets.getModule("thirdparty/lodash"),
        CommandManager  = brackets.getModule("command/CommandManager"),
        DocumentManager = brackets.getModule("document/DocumentManager"),
        EditorManager   = brackets.getModule("editor/EditorManager"),
        EditorModule    = brackets.getModule("editor/Editor"),
        FileSystem      = brackets.getModule("filesystem/FileSystem"),
        Git             = require("src/git/Git"),
        Utils           = require("src/Utils"),
        Preferences     = require("./Preferences"),
        DWGitEvents     = require("src/DWGit/DWGitEvents"),
        ErrorHandler    = require("src/ErrorHandler"),
        Promise         = require("bluebird");

    var gitAvailable = false,
        currentGitRoot = null,
        gutterName = "brackets-git-gutter",
        openWidgets = [];

    function clearWidgets() {
        openWidgets.forEach(function (mark) {
            var w = mark.lineWidget;
            if (w.visible) {
                w.visible = false;
                w.widget.clear();
            }
        });
        openWidgets = [];
    }

    function showGuttersAfterRevert(editor, mark, removedLineCount) {
        var cm = editor._codeMirror;
        if (!cm.gitGutters) {
            return;
        }

        // clear currently opened widgets
        clearWidgets();

        var lineCount = mark.lineCount - removedLineCount,
            startLine = mark.line;

        cm.clearGutter(gutterName);
        if (mark.type === "removed" || mark.type === "modified") {
            var markIndex = cm.gitGutters.indexOf(mark);
            if (markIndex !== -1) {
                cm.gitGutters.splice(markIndex, 1);
            }
        }

        var prevLine = startLine;

        // Remove marks of lines of modified block
        cm.gitGutters = _.filter(cm.gitGutters, function(obj) {
            return !obj.parentMark || obj.parentMark.line !== mark.line;
        });
        cm.gitGutters.forEach(function (obj) {
            if (obj.line >= startLine) {
                obj.line += lineCount;
            }
            var $marker = $("<div>")
            .addClass(gutterName + "-" + obj.type + " gitline-" + (obj.line + 1))
            .html("&nbsp;");
            cm.setGutterMarker(obj.line, gutterName, $marker[0]);
        });
    }

    function gutterClick(cm, lineIndex, gutterId) {
        if (!cm) {
            return;
        }

        if (gutterId !== gutterName && gutterId !== "CodeMirror-linenumbers") {
            return;
        }

        var mark = _.find(cm.gitGutters, function (o) { return o.line === lineIndex; });
        if (!mark || mark.type === "added") { return; }

        // we need to be able to identify cm instance from any mark
        mark.cm = cm;

        if (mark.parentMark) { mark = mark.parentMark; }

        var endLineNo = mark.line;

        if (mark.type === "modified") {
            var childMarks = _.filter(cm.gitGutters, function (o) { return o.parentMark && o.parentMark.line === mark.line });
            if (childMarks && childMarks.length) {
                endLineNo = childMarks[childMarks.length - 1].line;
            }
            endLineNo += 1;
        }

        if (!mark.lineWidget) {
            mark.lineWidget = {
                visible: false,
                element: $("<div class='" + gutterName + "-deleted-lines'></div>")
            };
            var $btn = $("<button/>")
                .addClass("brackets-git-gutter-copy-button")
                .text("R")
                .on("click", function () {
                    var doc = DocumentManager.getCurrentDocument();
                    doc.replaceRange(mark.content + "\n", {
                        line: mark.line,
                        ch: 0
                    }, {
                        line: endLineNo,
                        ch: 0
                    });
                    var curEditor = EditorManager.getActiveEditor();
                    showGuttersAfterRevert(curEditor, mark, endLineNo - mark.line);
                });
            $("<pre/>")
                .attr("style", "tab-size:" + cm.getOption("tabSize"))
                .text(mark.content || " ")
                .append($btn)
                .appendTo(mark.lineWidget.element);
        }

        if (mark.lineWidget.visible !== true) {
            mark.lineWidget.visible = true;
            mark.lineWidget.widget = cm.addLineWidget(mark.line, mark.lineWidget.element[0], {
                coverGutter: false,
                noHScroll: false,
                above: true,
                showIfHidden: false
            });
            openWidgets.push(mark);
        } else {
            mark.lineWidget.visible = false;
            mark.lineWidget.widget.clear();
            var io = openWidgets.indexOf(mark);
            if (io !== -1) {
                openWidgets.splice(io, 1);
            }
        }
    }

    function prepareGutter(editor) {
        // add our gutter if its not already available
        var cm = editor._codeMirror;

        var gutters = cm.getOption("gutters").slice(0);
        if (gutters.indexOf(gutterName) === -1) {
            gutters.unshift(gutterName);
            cm.setOption("gutters", gutters);
            cm.on("gutterClick", gutterClick);
        }
    }

    function showGutters(editor, _results) {
        prepareGutter(editor);

        var cm = editor._codeMirror;
        cm.gitGutters = _.sortBy(_results, "line");

        // clear currently opened widgets
        clearWidgets();

        cm.clearGutter(gutterName);
        cm.gitGutters.forEach(function (obj) {
            var $marker = $("<div>")
                            .addClass(gutterName + "-" + obj.type + " gitline-" + (obj.line + 1))
                            .html("&nbsp;");
            cm.setGutterMarker(obj.line, gutterName, $marker[0]);
        });
    }

    function processDiffResults(editor, diff) {
        var added = [],
            removed = [],
            modified = [],
            changesets = diff.split(/\n@@/).map(function (str) { return "@@" + str; });

        // remove part before first
        changesets.shift();

        changesets.forEach(function (str) {
            var m = str.match(/^@@ -([,0-9]+) \+([,0-9]+) @@/);
            var s1 = m[1].split(",");
            var s2 = m[2].split(",");

            // removed stuff
            var lineRemovedFrom;
            var lineFrom = parseInt(s2[0], 10);
            var lineCount = parseInt(s1[1], 10);
            if (isNaN(lineCount)) { lineCount = 1; }
            if (lineCount > 0) {
                lineRemovedFrom = lineFrom - 1;
                removed.push({
                    type: "removed",
                    line: lineRemovedFrom,
                    lineCount: lineCount,
                    content: str.split("\n")
                                .filter(function (l) { return l.indexOf("-") === 0; })
                                .map(function (l) { return l.substring(1); })
                                .join("\n")
                });
            }

            // added stuff
            lineFrom = parseInt(s2[0], 10);
            lineCount = parseInt(s2[1], 10);
            if (isNaN(lineCount)) { lineCount = 1; }
            var isModifiedMark = false;
            var firstAddedMark = false;
            for (var i = lineFrom, lineTo = lineFrom + lineCount; i < lineTo; i++) {
                var lineNo = i - 1;
                if (lineNo === lineRemovedFrom) {
                    // modified
                    var o = removed.pop();
                    o.type = "modified";
                    modified.push(o);
                    isModifiedMark = o;
                } else {
                    var mark = {
                        type: isModifiedMark ? "modified" : "added",
                        line: lineNo,
                        parentMark: isModifiedMark || firstAddedMark || null
                    };
                    if (!isModifiedMark && !firstAddedMark) {
                        firstAddedMark = mark;
                    }
                    // added new
                    added.push(mark);
                }
            }
        });

        // fix displaying of removed lines
        removed.forEach(function (o) {
            o.line = o.line + 1;
        });

        showGutters(editor, [].concat(added, removed, modified));
    }

    EditorModule.Editor.prototype.redrawGitGutterMarks = function () {
        if (!gitAvailable || !currentGitRoot) {
            return;
        }
        if(!_isMasterEditor(this))
            return;
        var currentFilePath = null;
        if (this.document && this.document.file) {
            currentFilePath = this.document.file.fullPath;
        }
        if (currentFilePath.indexOf(currentGitRoot) !== 0) {
            // file is not in the current project
            return;
        }
        var filename = currentFilePath.substring(currentGitRoot.length);
        var self = this;
        Git.dwDiffFileComplete(filename, currentGitRoot).then(function (diff) {
            processDiffResults(self, diff);
        }).catch(function (e) {
            if (ErrorHandler.contains(e, "Not a git repository")) {
                currentGitRoot = null;
            }
            var cm = self._codeMirror;
            if(cm.getOption("gutters").indexOf(gutterName) !== -1) {
                showGutters(self, []);
            }
        });
    };

    function reloadGutterMarks() {
        var activeEditor = EditorManager.getActiveEditor();
        if(!_isMasterEditor(activeEditor)) {
            return;
        }
        var cm = activeEditor._codeMirror;
        if(typeof cm.gitGutters === "undefined" || cm.getOption("gutters").indexOf(gutterName) === -1) {
            activeEditor.redrawGitGutterMarks();
        } else if (cm.gitGutters && cm.gitGutters.length > 0) {
            if (!currentGitRoot) {
                cm.gitGutters = [];
            }
            showGutters(activeEditor,cm.gitGutters);
        }
    }

    function goToPrev() {
        var activeEditor = EditorManager.getActiveEditor();
        if (!activeEditor) { return; }

        var results = activeEditor._codeMirror.gitGutters || [];
        var searched = _.filter(results, function (i) { return !i.parentMark; });

        var currentPos = activeEditor.getCursorPos();
        var i = searched.length;
        while (i--) {
            if (searched[i].line < currentPos.line) {
                break;
            }
        }
        if (i > -1) {
            var goToMark = searched[i];
            activeEditor.setCursorPos(goToMark.line, currentPos.ch);
        }
    }

    function goToNext() {
        var activeEditor = EditorManager.getActiveEditor();
        if (!activeEditor) { return; }

        var results = activeEditor._codeMirror.gitGutters || [];
        var searched = _.filter(results, function (i) { return !i.parentMark; });

        var currentPos = activeEditor.getCursorPos();
        for (var i = 0, l = searched.length; i < l; i++) {
            if (searched[i].line > currentPos.line) {
                break;
            }
        }
        if (i < searched.length) {
            var goToMark = searched[i];
            activeEditor.setCursorPos(goToMark.line, currentPos.ch);
        }
    }

    function getGitRoot(path) {
        return new Promise(function (resolve) {
            // keep .git away from file tree for now
            // this branch of code will not run for intel xdk
            if (typeof brackets !== "undefined" && brackets.fs && brackets.fs.stat) {
                brackets.fs.stat(path + ".git", function (err, result) {
                    var exists = err ? false : (result.isFile() || result.isDirectory());
                    if (exists) {
                        resolve(path);
                    } else {
                        resolve(null);
                    }
                });
                return;
            }
            FileSystem.resolve(path + ".git", function (err, item, stat) {
                var exists = err ? false : (stat.isFile || stat.isDirectory);
                if (exists) {
                    resolve(path);
                } else {
                    resolve(null);
                }
            });
        });
    }

    function _isMasterEditor(editor) {
        return editor && editor._paneId && editor.document && editor.document._masterEditor === editor;
    }

    function refreshGutterMarks(path, isGitRepo) {
        if(path && path.length !== 0) {
            if(!currentGitRoot && isGitRepo === "true") {// refresh gutter marks call for a file is only when it belongs to a git initialized site.
                currentGitRoot = Utils.getProjectRoot();
            }
            var doc = DocumentManager.getOpenDocumentForPath(path);
            if (doc) {
                doc.trigger("git-gutter-refresh");
            }
        } else {
            var docList = DocumentManager.getAllOpenDocuments();
             for(var i=0; i<docList.length; ++i) {
                 if(docList[i]) {
                     docList[i].trigger("git-gutter-refresh");
                 }
             }
        }
    }

    function dwSetGitPath(gitPath) {
        if(gitPath !== "false") {
            Preferences.set("gitPath", gitPath, null, true); // don't write the preference in brackets.json
            gitAvailable = true;
            refreshGutterMarks();
        } else {
            gitAvailable = false;
        }
    }
    /* // disable because of https://github.com/zaggino/brackets-git/issues/1019
    var _timer;
    var $line = $(),
        $gitGutterLines = $();

    $(document)
        .on("mouseenter", ".CodeMirror-linenumber", function (evt) {
            var $target = $(evt.target);

            // Remove tooltip
            $line.attr("title", "");

            // Remove any misc gutter hover classes
            $(".CodeMirror-linenumber").removeClass("brackets-git-gutter-hover");
            $(".brackets-git-gutter-hover").removeClass("brackets-git-gutter-hover");

            // Add new gutter hover classes
            $gitGutterLines = $(".gitline-" + $target.html()).addClass("brackets-git-gutter-hover");

            // Add tooltips if there are any git gutter marks
            if ($gitGutterLines.hasClass("brackets-git-gutter-modified") ||
                $gitGutterLines.hasClass("brackets-git-gutter-removed")) {

                $line = $target.attr("title", Strings.GUTTER_CLICK_DETAILS);
                $target.addClass("brackets-git-gutter-hover");
            }
        })
        .on("mouseleave", ".CodeMirror-linenumber", function (evt) {
            var $target = $(evt.target);

            if (_timer) {
                clearTimeout(_timer);
            }

            _timer = setTimeout(function () {
                $(".gitline-" + $target.html()).removeClass("brackets-git-gutter-hover");
                $target.removeClass("brackets-git-gutter-hover");
            }, 500);
        });
    */

    EditorManager.on("activeEditorChange",function() {
        var currentProjectRoot = Utils.getProjectRoot();
        if(currentGitRoot) {
            if(currentProjectRoot === currentGitRoot) {
                reloadGutterMarks();
                return;
            }
        }
        getGitRoot(currentProjectRoot).then(function (path) {
            if(path !== null) {
                currentGitRoot = path;
                reloadGutterMarks();
            } else {
                currentGitRoot = null;
            }
        });
    });

    function init() {
        DocumentManager.on("documentRefreshed", reloadGutterMarks);
        //register commands
        CommandManager.register(DWGitEvents.DW_GIT_REFRESH, DWGitEvents.DW_GIT_REFRESH, refreshGutterMarks);
        CommandManager.register(DWGitEvents.SET_GIT_PATH, DWGitEvents.SET_GIT_PATH, dwSetGitPath);
        dwbrackets.dwGitNotification(DWGitEvents.DW_GIT_CONFIG_PATH, "");
    }

    // API
    exports.goToPrev = goToPrev;
    exports.goToNext = goToNext;
    exports.init     = init;
});
