"use strict";
define(function (require, exports, module) {
    var _ = brackets.getModule('thirdparty/lodash');
    var DocumentManager = brackets.getModule('document/DocumentManager');
    var ExtensionUtils = brackets.getModule('utils/ExtensionUtils');
    var MainViewManager = brackets.getModule('view/MainViewManager');
    var EditorManager = brackets.getModule('editor/EditorManager');
    var GUTTER_NAME = 'brackets-inspection-gutter';
    var GUTTER_MARKER_NAME = 'brackets-inspection-gutter-marker';
    var GUTTER_WARNING_CLASS = 'brackets-inspection-gutter-warning';
    var GUTTER_ERROR_CLASS = 'brackets-inspection-gutter-error';
    var CM_LINE_NUMBER_GUTTER = 'CodeMirror-linenumbers';
    var markers = {};
    var editorsWithGutters = [];
    function prepareGutter(editor) {
        var cm = editor._codeMirror;
        var gutters = cm.getOption('gutters').slice(0);
        if (gutters.indexOf(GUTTER_NAME) === -1) {
            var cmLineNumberIdx = gutters.indexOf(CM_LINE_NUMBER_GUTTER);
            cmLineNumberIdx = cmLineNumberIdx === -1 ? 0 : cmLineNumberIdx;
            gutters.splice(cmLineNumberIdx, 0, GUTTER_NAME);
            cm.setOption('gutters', gutters);
        }
        if (editorsWithGutters.indexOf(editor) === -1) {
            editorsWithGutters.push(editor);
        }
    }
    function removeGutter(editor) {
        var cm = editor._codeMirror;
        if (!cm) {
            return;
        }
        var gutters = cm.getOption('gutters').slice(0);
        var io = gutters.indexOf(GUTTER_NAME);
        if (io !== -1) {
            gutters.splice(io, 1);
            cm.clearGutter(GUTTER_NAME);
            cm.setOption('gutters', gutters);
        }
        try {
            var fullPath = editor.document.file.fullPath;
            delete markers[fullPath];
        }
        catch (err) {
            console.error("Error clearing data from markers -> " + err);
        }
    }
    function prepareGutters(editors) {
        editors.forEach(function (editor) { return prepareGutter(editor); });
        var idx = editorsWithGutters.length;
        while (idx--) {
            if (editors.indexOf(editorsWithGutters[idx]) === -1) {
                removeGutter(editorsWithGutters[idx]);
                editorsWithGutters.splice(idx, 1);
            }
        }
    }
    function showGutters(editor, fullPath) {
        if (markers[fullPath] == null) {
            markers[fullPath] = {};
        }
        var markersForFile = Object.keys(markers[fullPath]).reduce(function (arr, sourceId) {
            return arr.concat(markers[fullPath][sourceId]);
        }, []);
        markersForFile = _.sortBy(markersForFile, function (obj) {
            switch (obj.type) {
                case 'problem_type_error':
                    return '1' + _.padLeft(obj.pos.line, 5, '0');
                case 'problem_type_warning':
                    return '2' + _.padLeft(obj.pos.line, 5, '0');
                case 'problem_type_meta':
                    return '3' + _.padLeft(obj.pos.line, 5, '0');
                default:
                    return '4' + _.padLeft(obj.pos.line, 5, '0');
            }
        });
        var lines = [];
        markersForFile = markersForFile.filter(function (obj) {
            if (lines.indexOf(obj.pos.line) === -1) {
                lines.push(obj.pos.line);
                return true;
            }
            return false;
        });
        var cm = editor._codeMirror;
        cm.clearGutter(GUTTER_NAME);
        markersForFile.forEach(function (obj) {
            var severity = obj.type === 'problem_type_error' ? GUTTER_ERROR_CLASS : GUTTER_WARNING_CLASS;
            var $marker = $('<div><span>')
                .attr('title', obj.message)
                .addClass(GUTTER_MARKER_NAME);
            $marker.find('span')
                .addClass(severity)
                .html('&nbsp;');
            var line = _.get(obj, 'pos.line') || 0;
            cm.setGutterMarker(line, GUTTER_NAME, $marker[0]);
        });
    }
    function set(sourceId, fullPath, report, options) {
        if (options === void 0) { options = true; }
        var errors = report.errors.filter(function (result) {
            if (result.type !== 'problem_type_error' &&
                result.type !== 'problem_type_warning' &&
                result.type !== 'problem_type_meta') {
                console.warn(sourceId + " -> Unexpected error type: " + result.type);
            }
            if (options === true) {
                return true;
            }
            if (options === false) {
                return false;
            }
            if (result.type === 'problem_type_error' && options.error !== true) {
                return false;
            }
            if (result.type === 'problem_type_warning' && options.warning !== true) {
                return false;
            }
            if (result.type === 'problem_type_meta' && options.meta !== true) {
                return false;
            }
            return true;
        });
        markers[fullPath] = markers[fullPath] || {};
        markers[fullPath][sourceId] = errors;
        var editors = _.compact(_.map(MainViewManager.getPaneIdList(), function (paneId) {
            var currentPath = MainViewManager.getCurrentlyViewedPath(paneId);
            var doc = currentPath && DocumentManager.getOpenDocumentForPath(currentPath);
            return doc && doc._masterEditor;
        }));
        prepareGutters(editors);
        var activeEditor = EditorManager.getActiveEditor();
        if (activeEditor && activeEditor.document === DocumentManager.getOpenDocumentForPath(fullPath)) {
            showGutters(activeEditor, fullPath);
        }
    }
    module.exports = function () {
        var w = window;
        if (w.bracketsInspectionGutters) {
            return;
        }
        ExtensionUtils.loadStyleSheet(module, '../styles/styles.less');
        w.bracketsInspectionGutters = { set: set };
    };
});
