// Copyright 2012-2013 Adobe Systems Incorporated.  All rights reserved.

//--------------------------------------------------------------------
// CLASS:
//   JQuery.DesignTime.Widget.Slider
//
// DESCRIPTION:
//   This class is used by the Slider property inspector to manage
//   and manipulate the design-time state of the widget's markup in the
//   design view.
//--------------------------------------------------------------------

//--------------------------------------------------------------------
// FUNCTION:
//   Slider
//
// DESCRIPTION:
//	 This is the constructor method for this design time Slider object.
//	 It calls the base widget constructor which helps us inherit all base methods
//	 like addClass, removeClass, ensureValidElements etc
//	 
//   It also manages the Slider's widget markup in the design view. This constructor
//   is registered in the JQuery Widget translator found at:
//
//       Configuration/Translators/jQueryWidget.htm
//	 
//
// ARGUMENTS:
//   dom - object - The DOM used by the document in design view.
//   element - object - The top-level DOM element for our widget markup.
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider = function(dom, element, consArgs)
{
  JQuery.DesignTime.Widget.Base.call(this, dom, element);

	this.opts = {};
	this.opts = consArgs;
  this.init(element);
};

// JQuery.DesignTime.Widget.Slider derives from our JQuery.DesignTime.Widget.Base
// class, so create a Base object and use it as our prototype so we inherit all of its
// methods.
JQuery.DesignTime.Widget.Slider.prototype = new JQuery.DesignTime.Widget.Base();

// Reset the constructor property back to Slider for our newly created prototype
// object so callers know that our object was created with the Slider constructor.
JQuery.DesignTime.Widget.Slider.prototype.constructor = JQuery.DesignTime.Widget.Slider;

//--------------------------------------------------------------------
// FUNCTION:
//   init
//
// DESCRIPTION:
//   Initializes the design-time object's properties. This  method is
//   called from the constructor and refresh() methods.
//
// ARGUMENTS:
//   element - object - The widget's top-level DOM element.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.init = function(element)
{
  this.element = this.getElement(element);
	this.rangeArray = [];
	this.animateArray = [];
	this.orientationArray = [];
	this.initRangeArray();
	this.initAnimateArray();
	this.initOrientationArray();
	
  this.canRefresh = true;
};

//--------------------------------------------------------------------
// FUNCTION:
//  	getMinValue
//
// DESCRIPTION:
//  	This method returns the value of the min attribute in the widget constructor.
//		If user changes the min value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the min attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getMinValue = function()
{
	if( this.opts && this.opts.min != null && typeof this.opts.min != 'undefined' )
	{
		return this.opts.min.toString();
	}
	return "";
}

JQuery.DesignTime.Widget.Slider.prototype.recalculateOpts = function()
{
	this.opts = {};
	var consArgs = this.getConstructorArgs(this.widgetType);
	this.opts = consArgs;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setMinValue
//
// DESCRIPTION:
//  	When a new value for min is attained for the min attribute from the PI
//		we have to update our opts object with this new min value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	NONE
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.setMinValue = function(minValue)
{	
	if( minValue == null || typeof minValue == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
	
	var oldMinValue = this.opts.min;
	
	if( minValue == "" || minValue == "0" || parseInt(minValue).toString() == 'NaN' )
	{
		this.opts.min = null;
	}	
	else
	{
		this.opts.min = parseInt(minValue);
	}
	
	if( oldMinValue != this.opts.min )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getRangeCheckValue
//
// DESCRIPTION:
//  	Returns true if range is defined. This means that we should check 
//		the range checkbox in the PI
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Boolean: Whether we should check the range checkbox
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getRangeCheckValue = function()
{
	if( this.opts && this.opts.range != null && typeof this.opts.range != 'undefined' )
	{
			return true;
	}
	return false;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setRangeCheckValue
//
// DESCRIPTION:
//  	Value of the range checkbox. If it is true, we set the range 
//		option to true. On false we set it to null signifying that 
//		range value should be removed from constructor
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.setRangeCheckValue = function(rangeCheckValue)
{
	if( rangeCheckValue == null || typeof rangeCheckValue == "undefined" )
		return;
	
	if (!this.opts)
		this.opts = {};
	
	var oldRangeValue = this.opts.range;
	
	if (rangeCheckValue)	
		this.opts.range = true;
	else
		this.opts.range = null;
		
	if( oldRangeValue != this.opts.range )	
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getAnimateCheckValue
//
// DESCRIPTION:
//  	Returns true if animate is defined. This means that we should check 
//		the range checkbox in the PI
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Boolean: Whether we should check the animte checkbox
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getAnimateCheckValue = function()
{
	if( this.opts && this.opts.animate != null && typeof this.opts.animate != 'undefined' )
	{
			return true;
	}
	return false;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setAnimateCheckValue
//
// DESCRIPTION:
//  	Argument is value of the animate checkbox. If it is true, we set the animate 
//		option to true. On false we set it to null signifying that 
//		animate value should be removed from constructor
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.setAnimateCheckValue = function(animateCheckValue)
{
	if( animateCheckValue == null || typeof animateCheckValue == "undefined" )
		return;
	
	if (!this.opts)
		this.opts = {};
	
	var oldAnimateValue = this.opts.animate;
	
	if (animateCheckValue)	
		this.opts.animate = true;
	else
		this.opts.animate = null;
		
	if( oldAnimateValue != this.opts.animate )	
		this.updateOptions(this.widgetType, this.opts);		
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getMaxValue
//
// DESCRIPTION:
//  	This method returns the value of the max attribute in the widget constructor.
//		If user changes the max value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the max attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getMaxValue = function()
{
	if( this.opts && this.opts.max != null && typeof this.opts.max != 'undefined' )
	{
		return this.opts.max.toString();
	}
	return "";
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setMaxValue
//
// DESCRIPTION:
//  	When a new value for max is attained for the max attribute from the PI
//		we have to update our opts object with this new max value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.setMaxValue = function(maxValue)
{	
	if( maxValue == null || typeof maxValue == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
	
	var oldMaxValue = this.opts.max;
		
	if( maxValue == "" || maxValue == "100" || parseInt(maxValue).toString() == 'NaN' )
	{
		this.opts.max = null;
	}	
	else
	{
		this.opts.max = parseInt(maxValue);
	}
	
	if( this.opts.max != oldMaxValue )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getStepValue
//
// DESCRIPTION:
//  	This method returns the value of the step attribute in the widget constructor.
//		If user changes the step value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the step attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getStepValue = function()
{
	if( this.opts && this.opts.step != null && typeof this.opts.step != 'undefined' )
	{
		return this.opts.step.toString();
	}
	return "";
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setStepValue
//
// DESCRIPTION:
//  	When a new value for step is attained for the step attribute from the PI
//		we have to update our opts object with this new step value and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.setStepValue = function(stepValue)
{	
	if( stepValue == null || typeof stepValue == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
		
	var oldStepValue = this.opts.step;	
		
	if( stepValue == "" || stepValue == "1" || parseInt(stepValue).toString() == 'NaN' )
	{
		this.opts.step = null;
	}	
	else
	{
		this.opts.step = parseInt(stepValue);
	}
	
	if( oldStepValue != this.opts.step )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getRangeListIndex
//
// DESCRIPTION:
//  	This method will return the index of the range List element
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of range list
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getRangeListIndex = function()
{
	if( this.opts && this.opts.range != null && typeof this.opts.range != 'undefined' )
	{
		if( this.opts.range === true )
			return 0;
			
		var rangeArray = this.getRangeArray();
		for( var i = 0; i < rangeArray.length; i++ )
		{
			if( this.opts.range.toString() == rangeArray[i] )
				return i;
		}
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setRangeListIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the range
//		list.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.setRangeListIndex = function(rangeIndex)
{
	if( rangeIndex == null || typeof rangeIndex == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
	
	var oldRangeValue = this.opts.range;
	if( rangeIndex == 0 )
		this.opts.range = true;
	else 
		this.opts.range = this.getRangeArray()[rangeIndex];
	
	if( this.opts.range != oldRangeValue )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getValue
//
// DESCRIPTION:
//  	This method returns the value of the value attribute in the widget constructor.
//		If user changes the value value in the code. This is where we 'read' that updated 
//		value from the object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	String: The value of the value attribute as string
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getValue = function()
{
	if( this.opts && this.opts.value != null && typeof this.opts.value != 'undefined' )
	{
		return this.opts.value;
	}
	if( this.opts && this.opts.values != null && typeof this.opts.values != 'undefined' )
	{
		return this.opts.values
	}
	return "";
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setValue
//
// DESCRIPTION:
//  	When a new value for value is attained for the value attribute from the PI
//		we have to update our opts object with this new value of value attribute and initiate an update cycle
//		for this change to be reflected in code.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.setValue = function(value)
{	
	if( value == null || typeof value == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
	
	var oldValue = this.opts.value;
	var oldValues = this.opts.values;
	if( value == "" || parseInt(value).toString() == 'NaN')
	{
		this.opts.value = null;
		this.opts.values = null;
	}	
	else
	{
		
		if( value.match(/,/g) )
		{
			try
			{
				eval('this.opts.values = [' + value + ']');
			}
			catch(e)
			{	
				if (dw.isDebugBuild())
					alert(e.message);
			}
			this.opts.value = null;
		}
		else
		{
			this.opts.value = parseInt(value);
			this.opts.values = null;
		}
	}
	
	if( oldValue != this.opts.value || oldValues != this.opts.values )
		this.updateOptions(this.widgetType, this.opts);		
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getAnimateListIndexOrValue
//
// DESCRIPTION:
//  	This method will return the index of the animate List element
//		or it may return the value of the newly entered number for animation
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of animate list or a new string value
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getAnimateListIndexOrValue = function()
{
	if( this.opts && this.opts.animate != null && typeof this.opts.animate != 'undefined' )
	{
		if( this.opts.animate === true )
			return 0;
				
		var animateArray = this.getAnimateArray();
		for( var i = 0; i < animateArray.length; i++ )
		{
			if( this.opts.animate.toString() == animateArray[i] )
				return i;
		}
		
		if( typeof this.opts.animate == 'number' )
		{
			this.animateArray.push(this.opts.animate.toString());
			return this.opts.animate.toString();
		}	
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setAnimateListIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the animate
//		list.
//
// ARGUMENTS:
//  	index of animate list
//
// RETURNS:
//  	None
//--------------------------------------------------------------------


JQuery.DesignTime.Widget.Slider.prototype.setAnimateListIndex = function(animateIndex)
{
	if( animateIndex == null || typeof animateIndex == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
	
	var oldAnimateValue = this.opts.animate;
	
	if( animateIndex == 0 )
		this.opts.animate = true;
	else 
		this.opts.animate = this.getAnimateArray()[animateIndex];
	
	if( oldAnimateValue != this.opts.animate )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setAnimateListValue
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the animate
//		list.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------


JQuery.DesignTime.Widget.Slider.prototype.setAnimateListValue = function(animateValue)
{
	if( animateValue == null || typeof animateValue == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
		
	var oldAnimateValue = this.opts.animate;	
	
	this.opts.animate = parseInt(animateValue);
	
	if( oldAnimateValue != this.opts.animate )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getOrientationListIndex
//
// DESCRIPTION:
//  	This method will return the index of the orientation List element
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	Number: The value of the index of orientation list
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getOrientationListIndex = function()
{
	if( this.opts && this.opts.orientation != null && typeof this.opts.orientation != 'undefined' )
	{
		var orientationArray = this.getOrientationArray();
		for( var i = 0; i < orientationArray.length; i++ )
		{
			if( this.opts.orientation.toString() == orientationArray[i] )
				return i;
		}
	}
	return 0;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	setOrientationListIndex
//
// DESCRIPTION:
//  	Here we set the opts object from the new index set in the orientation
//		list.
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	None
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.setOrientationListIndex = function(orientationIndex)
{
	if( orientationIndex == null || typeof orientationIndex == "undefined" )
		return;
		
	if (!this.opts)
		this.opts = {};
	
	var oldOrientationValue = this.opts.orientation;
	
	if( orientationIndex == 0 )
		this.opts.orientation = null;
	else
		this.opts.orientation = this.getOrientationArray()[orientationIndex];
	
	if( oldOrientationValue != this.opts.orientation )
		this.updateOptions(this.widgetType, this.opts);	
}

//--------------------------------------------------------------------
// FUNCTION:
//   getAssets
//
// DESCRIPTION:
//   Static function that returns the assets to be applied to page
//
// ARGUMENTS:
//   None
//
// RETURNS:
//   (array of objects)
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.getAssets = function()
{
	var assets = new Array();
	
	assetObject = new Object();
	assetObject.fullPath = jQSliderImagesPath;
	assetObject.file = jQSliderImagesFile;
	assetObject.type = "";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQCoreCssPath;
	assetObject.file = jQCoreCssFile;
	assetObject.type = "link";
	assets.push(assetObject);

	assetObject = new Object();
	assetObject.fullPath = jQCssThemePath;
	assetObject.file = jQCssThemeFile;
	assetObject.type = "link";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQSliderCssWidgetPath;
	assetObject.file = jQSliderCssWidgetFile;
	assetObject.type = "link";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQMainPath;
	assetObject.file = jQMainFile;
	assetObject.type = "javascript";
	assets.push(assetObject);
	
	assetObject = new Object();
	assetObject.fullPath = jQSliderJsPath;
	assetObject.file = jQSliderJsFile;
	assetObject.type = "javascript";
	assets.push(assetObject);
	
	return (assets);
	
};

//--------------------------------------------------------------------
// FUNCTION:
//   refresh
//
// DESCRIPTION:
//   None
//
// ARGUMENTS:
//   Syncs up the internal state of the design-time widget object
//   with the markup that currently exists in the design view.
//
//   This method gets called from the translator if a design-time
//   object already exists for a specific widget ID, and from various
//   methods in the Slider property inspector.
//
// RETURNS:
//   N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.refresh = function()
{
  if (!this.canRefresh)
	return;
	
	this.opts = {};
	var consArgs = this.getConstructorArgs(this.widgetType);
	this.opts = consArgs; 
  this.init(this.element_id);  
};


//--------------------------------------------------------------------
// FUNCTION:
//   JQuery.DesignTime.Widget.Slider.getNewSliderSnippet
//
// DESCRIPTION:
//   Static utility function that returns a string containing the
//   markup for a complete Slider.
//
// ARGUMENTS:
//  id - string - The string to use as the widget's id attribute.
//
// RETURNS:
//   String that is the HTML markup fragment for the panel.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.getNewSliderSnippet = function(id)
{
  var dpSnippet = '<div id="' + id + '"></div>';
  return dpSnippet;
};

//--------------------------------------------------------------------
// FUNCTION:
//   JQuery.DesignTime.Widget.Slider.getNewSliderConstructorSnippet
//
// DESCRIPTION:
//   Static utility function that returns a string that contains the
//   constructor snippet for creating a JQuery widget with the specified id.
//
// ARGUMENTS:
//  id - string - The id of the widget markup to insert into the
//                constructor snippet.
//
// RETURNS:
//   String containing JS that is the constructor call to create the
//   widget.
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.getNewSliderConstructorSnippet = function(id)
{
  return ' $(function() {\n\t$( "#' + id + '" ).slider(); \n});';
 /* $(function() {
		$( "#Slider1" ).slider();
	});*/
};

//--------------------------------------------------------------------
// FUNCTION:
//  	initRangeArray
//
// DESCRIPTION:
//  	Initializes the range array with predefined values. 
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.initRangeArray = function()
{
	var bothString = dw.loadString('jquery/widgets/slider/both');
	this.rangeArray.push(bothString);
	this.rangeArray.push("min");
	this.rangeArray.push("max");
}

//--------------------------------------------------------------------
// FUNCTION:
//  	initAnimateArray
//
// DESCRIPTION:
//  	Initializes the animate array with predefined values. 
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.initAnimateArray = function()
{
	var defaultString = dw.loadString('jquery/widgets/slider/default');
	this.animateArray.push(defaultString);
	this.animateArray.push("slow");
	this.animateArray.push("normal");
	this.animateArray.push("fast");
}

//--------------------------------------------------------------------
// FUNCTION:
//  	initOrientationArray
//
// DESCRIPTION:
//  	Initializes the orientation array with predefined values. 
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.initOrientationArray = function()
{
	this.orientationArray.push("horizontal");
	this.orientationArray.push("vertical");
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getRangeArray
//
// DESCRIPTION:
//  	Return the range Array containing all possible values for 'range'
//		attribute of the constructor object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getRangeArray = function()
{
	return this.rangeArray;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getAnimateArray
//
// DESCRIPTION:
//  	Return the animate Array containing all possible values for 'animate'
//		attribute of the constructor object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getAnimateArray = function()
{
	return this.animateArray;
}

//--------------------------------------------------------------------
// FUNCTION:
//  	getOrientationArray
//
// DESCRIPTION:
//  	Return the orientation Array containing all possible values for 'orientation'
//		attribute of the constructor object
//
// ARGUMENTS:
//  	None
//
// RETURNS:
//  	N/A
//--------------------------------------------------------------------

JQuery.DesignTime.Widget.Slider.prototype.getOrientationArray = function()
{
	return this.orientationArray;
}
