/*	
 Copyright 2015 Adobe Systems Incorporated.  All rights reserved. 

Purpose- 
This file contains the Responsive Adapter which invokes and creates the appropriate Responsive Framework
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50, continue: true */
/*global BootstrapFramework, ResponsiveFramework, ResponsiveLayoutConstants */

/* AdobePatentID="P5361-US". */

var ResponsiveAdapter = (
    function (ResponsiveFramework, BootstrapFramework) {
        "use strict";
        
        // define private variables here.
        var _isFrameworkSupported = false,
            _frameworkType = null,
            _responsiveFramework = null;

        /*
            function:ResponsiveAdapterImpl() - Constructor
            Arguments: frameworktype and the frameworkVersion of the responsive framework attached to the document
            Return: object.
        */
        var ResponsiveAdapterImpl = function (frameworkType, frameworkVersion) {
            if (this.identifyFrameworkType(frameworkType, frameworkVersion)) {
                this.constructResponsiveFramework(frameworkType, frameworkVersion);
            }
            if (!_isFrameworkSupported || !_responsiveFramework) {
                throw frameworkType + " Framework Type not supported";
            }
        };
        
        /*
            function:identifyFrameworkType() - Identify if there is a corresponding responsive framework that can be supported. Creates and Updates Data Strcutures if supported
            Arguments: frameworktype and the frameworkVersion of the responsive framework attached to the document
            Return: boolean.
        */
        ResponsiveAdapterImpl.prototype.identifyFrameworkType = function (frameworkType, frameworkVersion) {
            var property;
            for (property in ResponsiveLayoutConstants.FRAMEWORK) {
                if (ResponsiveLayoutConstants.FRAMEWORK[property] === frameworkType) {
                    _isFrameworkSupported = true;
                    _frameworkType = frameworkType;
                }
            }
            return _isFrameworkSupported;
        };

        /*
            function:constructResponsiveFramework() - Construct the supported responsive Framework type and version. Initialize all required models for UI.
            Arguments: frameworktype and the frameworkVersion of the responsive framework attached to the document
            Return: none.
        */
        ResponsiveAdapterImpl.prototype.constructResponsiveFramework = function (frameworkType, frameworkVersion) {
            //Create the appropriate ResponsiveFramework
            if (frameworkType === ResponsiveLayoutConstants.FRAMEWORK.BOOTSTRAP) {
                _responsiveFramework = new BootstrapFramework(frameworkVersion);
            } // else if { for other frameworks }
        };
        
        /*
            function:getUIModel() - Returns the UI models constructed by the Responsive Framework
            Arguments: none
            Return: UIModel
        */
        ResponsiveAdapterImpl.prototype.getUIModel = function () {
            return _responsiveFramework.getUIModel();
        };

        /*
            function:getGridModel() - Returns the Grid model constructed by the Responsive Framework
            Arguments: none
            Return: GridModel.
        */
        ResponsiveAdapterImpl.prototype.getGridModel = function () {
            return _responsiveFramework.getGridModel();
        };
        
        /*
            function:supportsFeature() - Returns the state of support of feature available
            Arguments: featureID
            Return: Boolean.
        */
        ResponsiveAdapterImpl.prototype.supportsFeature = function (featureID) {
            return _responsiveFramework.supportsFeature(featureID);
        };
        
        /*
            function:rediscoverElements() - Rediscover the Responsive elements in the document
            Arguments:
            Return:
        */
        ResponsiveAdapterImpl.prototype.rediscoverElements = function (inUnhideMode) {
            return _responsiveFramework.rediscoverElements(inUnhideMode);
        };

        /*
            function:hideElement() - Hide the given Element
            Arguments: DOM Element to be hidden
            Return:
        */
        ResponsiveAdapterImpl.prototype.hideElement = function (domElement) {
            return _responsiveFramework.hideElement(domElement);
        };

        /*
            function:unhideElement() - Unhide the given Element
            Arguments: DOM Element to be shown
            Return:
        */
        ResponsiveAdapterImpl.prototype.unhideElement = function (domElement) {
            return _responsiveFramework.unhideElement(domElement);
        };
        
        /*
            function:addNewRow() - Add new row adjacent to the current selection
            Arguments: Current DOM Element
            Return:
        */
        ResponsiveAdapterImpl.prototype.addNewRow = function (domElement) {
            return _responsiveFramework.addNewRow(domElement);
        };
        
        /*
            function:addNewColumn() - Add new column adjacent to the current selection
            Arguments: Current DOM Element
            Return:
        */
        ResponsiveAdapterImpl.prototype.addNewColumn = function (domElement) {
            return _responsiveFramework.addNewColumn(domElement);
        };
        
        /*
            function:constructGridForActiveElement() - construct the grid for the current element
            Arguments: Current DOM Element being edited
            Return: currentGrid: {
                        left: -1,
                        right: -1,
                        width: -1,
                        numCols: -1,
                        columns: [], // [leftOfCol, ...]
                        columnWidth: [] // [1 column width, 2 columnWidth]
                    }
        */
        ResponsiveAdapterImpl.prototype.constructGridForActiveElement = function (domElement) {
            return _responsiveFramework.constructGridForActiveElement(domElement);
        };

        /*
            function:resizeTo() - Resize the passed element to the number of columns specified.
            Arguments: Current DOM Element being edited, new columns size, commit to Dw
            Return:
        */
        ResponsiveAdapterImpl.prototype.resizeTo = function (domElement, newColumn, commit) {
            _responsiveFramework.resizeTo(domElement, newColumn, commit);
        };
        
        /*
            function:shiftTo() - Resize the offset of passed element to the new column specified.
            Arguments: Current DOM Element being edited, new column indicating offset position, commit to Dw
            Return:
        */
        ResponsiveAdapterImpl.prototype.shiftTo = function (domElement, newColumn, commit) {
            _responsiveFramework.shiftTo(domElement, newColumn, commit);
        };
        
        /*
            function: getAllInfoForResponsiveInsert - required while Inserting
            Arguments: none
            Return: All the class names to be applied, read from the xml file.
        */
        
        ResponsiveAdapterImpl.prototype.getAllInfoForResponsiveInsert = function () {
            return _responsiveFramework.getAllInfoForResponsiveInsert();
        };
		
		/*
            function: getGridSizesWithPrefix - For adding menu items to Scrubber context menu and Window Size menu for Bootstrap pages
            Arguments: none
            Returns: Bootstrap breakpoints value with the size prefix as a string eg. xs-0-767;sm-768-991
        */
        ResponsiveAdapterImpl.prototype.getGridSizesWithPrefix = function () {
            return _responsiveFramework.getGridSizesWithPrefix();
        };

        /*
            function: isAValidColumn - checks if the passed element is actually a column
            Arguments: element to be checked
            Return: bool
        */
        ResponsiveAdapterImpl.prototype.isAValidColumn = function (colElem) {
            return _responsiveFramework.isAValidColumn(colElem);
        };

        /*
            function: isAValidRow - checks if the passed element is actually a row
            Arguments: element to be checked
            Return: bool
        */
        ResponsiveAdapterImpl.prototype.isAValidRow = function (rowElem) {
            return _responsiveFramework.isAValidRow(rowElem);
        };
        
        /*
            function: canHideElement - required for context menu. Verifies if the element is contained within a hidden element or not
            Arguments: element to be checked
            Return: bool
        */
        ResponsiveAdapterImpl.prototype.canHideElement = function (domElement) {
            return _responsiveFramework.canHideElement(domElement);
        };
        
        /*
            function: isElementHidden - required for context menu. Verifies if the element is hidden
            Arguments: element to be checked
            Return: bool
        */
        ResponsiveAdapterImpl.prototype.isElementHidden = function (domElement) {
            return _responsiveFramework.isElementHidden(domElement);
        };
        
        return ResponsiveAdapterImpl;
    }(ResponsiveFramework, BootstrapFramework)
);