/*
 Copyright 2019 Adobe Systems Incorporated.  All rights reserved.
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */

class RangeDeleteHandler extends BaseEventHandler {

    constructor(range, forward, copyToClipboard) {
        'use strict';

        super();

        this.m_forward = forward;
        this.m_copyToClipboard = copyToClipboard;

        this.m_range = range.cloneRange();
        this.m_postOpIp = this.m_range.cloneRange(); // Selection post-delete
        this.m_postOpIp.collapse(!this.m_forward);

        this.m_start = range.startContainer.childNodes[range.startOffset]
                || range.startContainer; //it's a text node
        this.m_end = range.endContainer.childNodes[range.endOffset]
                || range.endContainer; //it's a text node

        if(this.m_start.nodeType === Node.TEXT_NODE)
            this.m_start = this.m_start.parentNode;
        if(this.m_end.nodeType === Node.TEXT_NODE)
            this.m_end = this.m_end.parentNode;
    }

    onStart(event) {
        'use strict';

        if (event) {
            event.preventDefault();
            event.stopPropagation();
        }

        // Copy to clipboard
        if(this.m_copyToClipboard)
            this._copy();
 
        // Delete content
        this.m_range.deleteContents();

        // Do post processing
        this._postProcessing();
    }

    onEnd(event) {
        'use strict';

        if (event) {
            event.preventDefault();
            event.stopPropagation();
        }
    }

    getEditRange(event) {
        'use strict';
        return this.m_range;
    }

    getAffectedElement() {
        'use strict';
        return super.getAffectedElement();
    }

    _copy() {
        'use strict';

        var liveEditEvent = window.parent.globalController.getLiveEditEventObject();
        if(liveEditEvent) {
            liveEditEvent.fire({type: parent.window.DW_LIVEEDIT_EVENTS.CopySelectionToClipboard});
        }
    }

    _postProcessing() {
        'use strict';

        // Make sure that end containers are non-empty
        this._makeNonEmptyEndContainers();

        // Set the selection
        this._setSelection();
    }

    _makeNonEmptyEndContainers() {
        'use strict';

        if(this.m_start.isConnected && parent.DWTextEditUtility.isElementEmpty(this.m_start)) {
            var textNode = parent.document.createTextNode( String.fromCharCode( parent.DW_LIVEEDIT_CONSTANTS.NBSPCharCode));
            this.m_start.appendChild(textNode);
        }

        if(this.m_end.isConnected && this.m_start !== this.m_end && parent.DWTextEditUtility.isElementEmpty(this.m_end)) {
            var textNode = parent.document.createTextNode( String.fromCharCode( parent.DW_LIVEEDIT_CONSTANTS.NBSPCharCode));
            this.m_end.insertBefore(textNode, this.m_end.firstChild);
        }
    }

    _setSelection() {
        'use strict';

        var selection = parent.window.getSelection();
        if(!selection)
            return;

        selection.removeAllRanges();
        selection.addRange(this.m_postOpIp);
    }
}