/*
 Copyright 2019 Adobe Systems Incorporated.  All rights reserved.
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */

class DeleteHandler extends BaseEventHandler {

    constructor(forward, copyToClipboard) {
        'use strict';

        super();
        this.m_forward = forward || false;
        this.m_copyToClipboard = copyToClipboard || false;
        this.m_deleter = this._getDeleter();
    }

    onStart(event) {
        'use strict';

        if(this.m_deleter) {
            this.m_deleter.onStart(event);
        } else if (event) {
            event.preventDefault();
            event.stopPropagation();
        }
    }

    onEnd(event) {
        'use strict';

        if(this.m_deleter) {
            this.m_deleter.onEnd(event);
        } else if (event) {
            event.preventDefault();
            event.stopPropagation();
        }
    }

    getAffectedElement() {
        'use strict';

        if(this.m_deleter)
            return this.m_deleter.getAffectedElement();
        return null;
    }

    getEditRange(event) {
        'use strict';

        if(this.m_deleter)
            return this.m_deleter.getEditRange(event);
        return null;
    }

    _getDeleter() {
        'use strict';

        var deleter = null;
        var selection = parent.window.getSelection();
        if(!selection || selection.rangeCount !== 1)
            return null;

        var range = null;
        if(selection.isCollapsed) {
            var editability = this._getEditability();
            if(editability.isEditable && !editability.isAcrossTextSessions)
                range = editability.editRange;
        } else {
            range = selection.getRangeAt(0);
        }

        if(range)
            deleter = new RangeDeleteHandler(range, this.m_forward, this.m_copyToClipboard);

        return deleter;
    }

    // Get editability of next text offset. It goes though all nodes starting from the focusNode
    // to find any text offset which can be edited. If while finding this offset, we come across
    // comment or dynamically generated element then offset is non-editable.
    _getEditability() {
        'use strict';

        var isEditable = false;
        var editRange = null;
        var isAcrossTextSessions = false;
        var selection = parent.window.getSelection();

        if(selection && selection.focusNode) {

            var focusNode = selection.focusNode;
            var focusOffset = selection.focusOffset;
            var forward = this.m_forward;

            do {
                // 1. Check if focusNode is text node and it has non-whitespace chars
                if(focusNode.nodeType === Node.TEXT_NODE) {
                    editRange = this._getNonWsRange(focusNode, focusOffset, forward);
                    if(editRange) {
                        isEditable = true;
                        break;
                    }
                }

                var container = focusNode.childNodes[focusOffset] || focusNode;
                var root = super.getAffectedElement();
                var iterator = forward ? new ForwardIterator(container, root) : new BackwardIterator(container, root)
                var node = null;
                var stop = false;

                // Iterate through nodes to look for comments
                while((node = iterator.next()) !== null) {
                    switch(node.nodeType) {
                        // If comment is next then its not editable
                        case Node.COMMENT_NODE:
                            stop = true;
                            break;
                        // If text node has some non-whitespace text then it has editable offset
                        case Node.TEXT_NODE:
                            // Ignore text node, if already covered in step 1.
                            // Or if text node is specified as offset in parent, then resultant caret
                            // would be at the beginning of the text node. Hence, it should be
                            // considered only in case of 'forward' and ignore for 'backward'.
                            if(node === focusNode || (!forward && node === container))
                                break;
                            editRange = this._getNonWsRange(node, -1, forward);
                            if(editRange) {
                                isEditable = true;
                                stop = true;
                            }
                            break;
                        // Check for elements
                        case Node.ELEMENT_NODE:
                            if(!node.getAttribute( parent.DW_LIVEEDIT_CONSTANTS.DWUniqueId)) {
                                // If element is generated dynamically then its non-editable
                                stop = true;
                            } else if(node.tagName === "BR") {
                                // BR tag is editable
                                editRange = parent.document.createRange();
                                editRange.setStartBefore(node);
                                editRange.setEndAfter(node);

                                isEditable = true;
                                stop = true;
                            } else {
                                // Only inline elements form a text-session
                                if(!parent.DW_LIVE_TEXT_EDIT.InlineTextElements.includes(node.tagName))
                                    isAcrossTextSessions = true;
                            }
                            break;
                    }

                    if(stop)
                        break;
                }
            } while(false);

            // Edit range can be across text sessions if,
            // 1. any block level element comes during edit range calculation (determined above)
            // 2. there exists block level ancestor from focus node or edit range node till their
            //    common ancestor container
            if(isEditable && !isAcrossTextSessions) {

                // Calculate full range
                var fullRange = parent.document.createRange();
                if(forward) {
                    fullRange.setStart(focusNode, focusOffset);
                    fullRange.setEnd(editRange.endContainer, editRange.endOffset);
                } else {
                    fullRange.setStart(editRange.startContainer, editRange.startOffset);
                    fullRange.setEnd(focusNode, focusOffset);
                }

                // Check if there exists any block-level element till their common ancestor
                var start = fullRange.startContainer.childNodes[fullRange.startOffset]
                        || fullRange.startContainer; //it's a text node
                var end = fullRange.endContainer.childNodes[fullRange.endOffset]
                        || fullRange.endContainer; //it's a text node

                // Start container
                while(!isAcrossTextSessions && start && start !== fullRange.commonAncestorContainer) {
                    if(start.nodeType === Node.ELEMENT_NODE && !parent.DW_LIVE_TEXT_EDIT.InlineTextElements.includes(start.tagName)) {
                        isAcrossTextSessions = true;
                    }
                    start = start.parentNode;
                }

                // End container
                while(!isAcrossTextSessions && end && end !== fullRange.commonAncestorContainer) {
                    if(end.nodeType === Node.ELEMENT_NODE && !parent.DW_LIVE_TEXT_EDIT.InlineTextElements.includes(end.tagName)) {
                        isAcrossTextSessions = true;
                    }
                    end = end.parentNode;
                }
            }
        }

        var editability = {};
        editability.isEditable = isEditable;
        editability.editRange = editRange;
        editability.isAcrossTextSessions = isAcrossTextSessions;
        return editability;
    }

    // Get range of first/last non-whitespace char from given offset.
    // If offset is -1 then entire text of node is considered.
    _getNonWsRange(node, offset, forward) {
        'use strict';

        var editRange = null;
        var offsetLast = -1;
        if(forward) {
            offset = TextUtils.firstNonWsCharIndex(node.nodeValue, offset);
            offsetLast = offset + 1;
        } else {
            offsetLast = TextUtils.lastNonWsCharIndex(node.nodeValue, offset);
            offset = offsetLast - 1;
        }
        if(offset >= 0 && offsetLast > 0) {
            editRange = parent.document.createRange();
            editRange.setStart(node, offset);
            editRange.setEnd(node, offsetLast);
        }
        return editRange;
    }
}