/*
 Copyright 2013 Adobe Systems Incorporated.  All rights reserved.

Purpose-
This file has the implementation of Image Editing functionality in Live Edit view
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global $, liveViewObject, dwObject, liveExtensionsConfigPath, dwExtensionController, KEYCODES ,DW_CONSTANTS,TAGS, DW_EXTENSION_EVENT*/

function AuxiliaryExtensionObject() {
    'use strict';
    
    this.m_currentSelectedElement = null;
    this.m_editable = true;
    
    this.m_currentHudHeight = 0; // current Hud Height
    this.m_currentHudWidth = 0; // current Hud width
    this.disableDrag = false; // Disable Dragging when mouse is out of Browser Window
    this.m_isShown = false;
    this.m_hudDragged = false;
    this.setCurrentHudTag("");
    
    this.eshDetails = null;

}

var DWLE_DIMENSIONS = {
    AuxHudOffSet: 13
};

AuxiliaryExtensionObject.prototype.getExtensionId = function () {
    "use strict";
    console.log("base class does not contain the id");
    return "";
};

// setter function to set if element is editable or not
AuxiliaryExtensionObject.prototype.setEditable = function (isEditable) {
    'use strict';
    this.m_editable = isEditable;
};

AuxiliaryExtensionObject.prototype.getEditable = function () {
    'use strict';
    return this.m_editable;
};

AuxiliaryExtensionObject.prototype.getHudTag = function () {
    'use strict';
    return "";
};

AuxiliaryExtensionObject.prototype.getCurrentHudTag = function () {
    'use strict';
    return this.currenthudTag;
};

AuxiliaryExtensionObject.prototype.setCurrentHudTag = function (tagName) {
    'use strict';
    this.currenthudTag = tagName;
};


AuxiliaryExtensionObject.prototype.toggleAuxHud = function (message) {
    'use strict';
    if (message.eshDetails && this.getHudTag().indexOf(message.eshDetails.Tag) !== -1) {
        if (this.getVisibility() === false) {
            //set current HUD essentials
            this.setCurrentHudTag(message.eshDetails.Tag);
            this.setVisibility(true);
            
            this.m_currentSelectedElement = liveViewObject.getCurrentSelectedElement();
            this.setEditable(liveViewObject.isElementEditable(this.m_currentSelectedElement));
            this.eshDetails = message.eshDetails;
            
            // enable the extension
            var extensionID = this.getExtensionId();
            liveViewObject.positionExtensionById(extensionID, 0, 0, parent.innerWidth + parent.scrollX, parent.innerHeight + parent.scrollY);
            liveViewObject.showExtensionById(extensionID);
            
            //show the content
            this.populateContent();
            this.positionHud();
            
        } else if (this.getVisibility() === true) {
            message.commit = true;
            this.hideAuxHud(message);
            this.setVisibility(false);
            this.setCurrentHudTag("");
        }
        this.m_hudDragged = false;
        return true;
    }
    return false;
};

AuxiliaryExtensionObject.prototype.shouldHide = function () {
    'use strict';
    return true;
};

AuxiliaryExtensionObject.prototype.hideAuxHud = function (message) {
    'use strict';
    var messageDetails;
    if (this.getVisibility() === true && this.shouldHide()) {
        
        if (message.commit === true) {
            // commit edits before hiding
            this.commit();
        }
        
        liveViewObject.hideExtensionById(this.getExtensionId());
        this.setVisibility(false);
        this.setCurrentHudTag("");
        this.m_hudDragged = false;
        
        if (this.getExtensionId() === "dwImageHud") {
            this.removeHudListeners();
        }
        
        if (document.activeElement) {
            // If any element in the Hud as focus we should remove it.
            document.activeElement.blur();
        }
        liveViewObject.clearBrowserUndoRedos();
    }
    if (!message.keepESHOpen) {
        messageDetails = {type: DW_EXTENSION_EVENT.RELOAD_ESH};
        if (message && typeof message.shouldScroll !== "undefined") {
            messageDetails.shouldScroll = message.shouldScroll;
        }
        dwExtensionController.sendMessage(messageDetails);
    }
    
};

AuxiliaryExtensionObject.prototype.fileSave = function () {
    'use strict';
    console.log("AuxiliaryExtensionObject::populateContent**Does not have enough information to Save the file. This method should be overriden by the extension to commit changes and save the file on callback**");
};


AuxiliaryExtensionObject.prototype.fileSaveAs = function () {
    'use strict';
    console.log("AuxiliaryExtensionObject::populateContent**Does not have enough information to Save As the file. This method should be overriden by the extension to commit changes and save the file on callback**");
};

AuxiliaryExtensionObject.prototype.messageHandler = function (e) {
    'use strict';
    if (e.data && e.data.type) {
        if (e.data.type === DW_EXTENSION_EVENT.THEME_CHANGED && window.setCoralThemeForElement) {
            this.m_currentTheme = e.data.theme;
            window.setCoralThemeForElement(this.m_hudContentDiv, e.data.theme);
        }
    }
};

AuxiliaryExtensionObject.prototype.initialize = function () {
    'use strict';
    this.setVisibility(false);
    this.setCurrentHudTag("");
    this.m_hudContainer = document.body;
    dwExtensionController.addListener(DW_EXTENSION_EVENT.FILE_SAVE, this.fileSave, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.FILE_SAVE_AS, this.fileSaveAs, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.HIDE_AUX_HUD, this.hideAuxHud, this);
    dwExtensionController.addListener(DW_EXTENSION_EVENT.TOGGLE_AUX_HUD, this.toggleAuxHud, this);
    
        //add key down handler to handle escape key
    var keyDownHandler = function (evt) {
        evt = evt || window.event;
        if (evt.keyCode === KEYCODES.Escape) {
            this.escapeKeyPressed();
        }
    };
    
    var keyUpHandler = function (evt) {
        evt = evt || window.event;
        if (evt.keyCode === KEYCODES.Delete || evt.keyCode === KEYCODES.BackSpace) {
            this.deleteOrbackspaceKeyPressed();
        }
    };
    this.m_currentTheme = liveViewObject.currentTheme;
    if (window.setCoralThemeForElement) {
        window.setCoralThemeForElement(this.m_hudContentDiv, liveViewObject.currentTheme);
    }
    window.addEventListener("keydown", keyDownHandler.bind(this), false);
    window.addEventListener("keyup", keyUpHandler.bind(this), false);
    window.addEventListener("message", this.messageHandler.bind(this), false);
};

AuxiliaryExtensionObject.prototype.getCurrentElement = function () {
    'use strict';
    return this.m_currentSelectedElement;
};
AuxiliaryExtensionObject.prototype.elementDimensionChanged = function () {
    'use strict';
    var messageDetails = {};
    messageDetails.type = DW_EXTENSION_EVENT.ELEMENT_DIMENSION_CHANGED;
    dwExtensionController.sendMessage(messageDetails);
};
AuxiliaryExtensionObject.prototype.escapeKeyPressed = function () {
    'use strict';
    if (!liveViewObject.isExtensionVisible(this.getExtensionId())) {
        return;
    }
    var message = {};
    message.commit = false;
    this.hideAuxHud(message);
};

AuxiliaryExtensionObject.prototype.deleteOrbackspaceKeyPressed = function () {
    'use strict';
    window.parent.writeDWLog("AuxiliaryExtensionObject::deleteOrbackspaceKeyPressed**This method should be overriden by the extension to handle delete or backspace key");
};

AuxiliaryExtensionObject.prototype.setVisibility = function (show) {
    'use strict';
    this.m_isShown = show;
    window.parent.liveViewExtensionsObject.setAuxHudVisibility(this.getCurrentHudTag(), this.getExtensionId(), show);
};

AuxiliaryExtensionObject.prototype.getVisibility = function () {
    'use strict';
    return this.m_isShown;
};

AuxiliaryExtensionObject.prototype.populateContent = function () {
    'use strict';
    console.log("AuxiliaryExtensionObject::populateContent**Does not have enough information to populate any content. This method should be overriden by the extension to generate the content/UI**");
};

AuxiliaryExtensionObject.prototype.commit = function () {
    'use strict';
    console.log("AuxiliaryExtensionObject::Commit**Does not have enough information to commit. This method should be overriden by the extension to commit the changes**");
};

AuxiliaryExtensionObject.prototype.positionHud = function () {
    'use strict';
    console.log("AuxiliaryExtensionObject::positionHud**Does not have enough information to position HUD. This method should be overriden by the extension to position itself properly**");
};

AuxiliaryExtensionObject.prototype.EnableDragOnElement = function (element) {
    'use strict';
    element.addEventListener("mousedown", function (event) {
        this.disableDrag = false;
        if (event.target && event.target.tagName !== TAGS.Input && event.target.tagName !== TAGS.TextArea && event.target.tagName !== TAGS.Select) {
            this.dragTheHud(this.m_hudContentDiv, event);
            window.addEventListener("mouseout", this.mouseOutHandler.bind(this));
        }
    }.bind(this), true);
};

/**
 * @private dragTheHud
 * This function is used for dragging the Hud around the browser surface
 * whenever mouse goes out of the browser we need to remove mouse move listerners to avoid ghost effect.
 * @Param elementToDrag  element being dragged
 * @Param event : mousedown event 
 */
AuxiliaryExtensionObject.prototype.dragTheHud = function (elementToDrag, event) {
    'use strict';
    var deltaX = event.clientX - parseInt(elementToDrag.style.left, 10);
    var deltaY = event.clientY - parseInt(elementToDrag.style.top, 10);
    document.body.style.webkitUserSelect = "none";
    var auxHudContext = this;
	/*when mouse move happens keep updating the hud position*/
    var moveHandler = function (e) {
        if (!e) {
            e = window.event;
        }
        if (auxHudContext.disableDrag === true) {
			// this means mouse has gone outside the browser window. So remove the handlers for mousemove Otherwise ghosting effect will be there
            if (elementToDrag.removeEventListener) {
                window.removeEventListener("mouseout", this.mouseOutHandler.bind(this));
                document.removeEventListener("mousemove", moveHandler, true);
                document.body.style.webkitUserSelect = "all";
            }
            return;
        }
		// if the new position is inside the browser window then update the position of the HUD
        var newPosX = e.clientX - deltaX;
        var newPosY = e.clientY - deltaY;
        var rightSide = window.innerWidth + window.scrollX - auxHudContext.m_currentHudWidth - DWLE_DIMENSIONS.AuxHudOffSet;
        var bottomSide = window.innerHeight + window.scrollY - auxHudContext.m_currentHudHeight - DWLE_DIMENSIONS.AuxHudOffSet;
        if (newPosX > DWLE_DIMENSIONS.AuxHudOffSet && newPosX < rightSide && newPosY > DWLE_DIMENSIONS.AuxHudOffSet && newPosY < bottomSide) {
            elementToDrag.style.left = newPosX + "px";
            elementToDrag.style.top = newPosY + "px";
        }
        if (e.stopPropagation) {
            e.stopPropagation();
        } else {
            e.cancelBubble = true;
        }
    };
	/*when mouse up happens dragging should stop*/
    var upHandler = function (e) {
        if (!e) {
            e = window.event;
        }
        if (elementToDrag.removeEventListener) {
            document.removeEventListener("mousemove", moveHandler, true);
            document.body.style.webkitUserSelect = "all";
            document.removeEventListener("mouseup", upHandler, true);
            window.removeEventListener("mouseout", auxHudContext.mouseOutHandler.bind(this));
        }
        auxHudContext.m_hudDragged = true;
    };

    if (document.addEventListener) {
        document.addEventListener("mousemove", moveHandler, true);
        document.addEventListener("mouseup", upHandler, true);
    }

};
    
AuxiliaryExtensionObject.prototype.mouseOutHandler = function (e) {
    'use strict';
    this.disableDragging(e);
};

/**
 * @private disableDragging
 * Whenever mouse leaves browser window Disable Dragging of the Image HUD
 * @Param event     : on mouse out comes sometimes comes when we are inside the browser also
 * So need to screen the occurrences.
 */
AuxiliaryExtensionObject.prototype.disableDragging = function (event) {
    'use strict';
    event = event || window.event;
    var toElement = event.relatedTarget || event.toElement;
    if (!toElement) {
        this.disableDrag = true;
    }
};

AuxiliaryExtensionObject.prototype.setIgnoreSelectionChange = function () {
    'use strict';
    var messageDetails = {};
    messageDetails.type = DW_EXTENSION_EVENT.IGNORE_SELECTION_CHANGE;
    dwExtensionController.sendMessage(messageDetails);
};

AuxiliaryExtensionObject.prototype.removeHudListeners = function () {
    "use strict";
    console.log("base class does not remove Listeners");
    return "";
};

