/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2013-2014 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, plusplus: true, devel: true, nomen: true, indent: 4, bitwise: true */
/*global define: true, graphite: true, Modernizr: true, window: true, FileReader: true */

define([
    'jquery',
    'underscore',
    'backbone',
    'plugin-dependencies',
    'views/landing/AssetView',
    'controllers/UploadController',
    'controllers/AuthController',
    'controllers/AssetListController',
    'utils/TemplateUtil'
], function ($, _, Backbone, deps, AssetView, UploadController,
    AuthController, AssetListController, TemplateUtil) {

    "use strict";

    var AssetListView = Backbone.View.extend({

        el: "#results",
        collection: null,
        loading: true,
        childViews: [],

        tagName: "section",

        initialize: function (authModel, path) {
            // get Asset List
            this.collection = AssetListController.getAssetList(
                this.onAssetCollectionSuccess,
                this.onAssetCollectionError,
                this,
                path
            );
            graphite.events.on('update-asset-list-view', function () {
                this.showUploadGuidance();
            }, this);
            this.addHandlers();
            this.render();
        },

        addHandlers: function () {
            this.collection.on("add", this.renderNewAsset, this);
            this.collection.on("reset", this.onReset, this);

            var viewThis = this,
                clickHandler = function (e) {
                    viewThis.childViews.forEach(function (view) {
                        view.stopListening();
                    });
                };

            graphite.events.on('upload-init', this.uploadFiles, this);
            $('.logo').on('click.logoClick', clickHandler);

            graphite.events.on('route-change', function (newRoute) {
                // clean up
                if (newRoute === 'handleAsset') {
                    graphite.events.off('upload-init', viewThis.uploadFiles);
                    $('.logo').off('click.logoClick', clickHandler);

                    viewThis.childViews.forEach(function (view) {
                        view.stopListening();
                    });
                }
            });
        },

        removeEventListeners: function () {
            $('.logo').off('.logoClick');
            this.collection.off(null, null, this);
            graphite.events.off(null, null, this);
        },

        //------------------------------------------------
        // Handlers
        //------------------------------------------------
        onAssetCollectionSuccess: function (e) {
            if (this.loading) {
                this.onReset();
                this.checkFolderSharing(e);
            }
        },

        onAssetCollectionError: function (response) {
            if (response.status === 401) {
                graphite.events.trigger("reauthenticate");
            }

            if ((null === this.scErrorText || undefined === this.scErrorText || "" === this.scErrorText) && response.status >= 500) {
                this.scErrorText = "<em>SharedCloud Services currently unavailable.  Please try again</em>";
                this.scErrorText = this.scErrorText + "<br/><br/>(HTTP status code: " + response.status + ")";
            }
            if (this.loading) {
                this.onReset();
            }
        },


        onReset: function () {
            this.loading = false;
            this.render();
            UploadController.restoreUploads(this.collection);
        },

        //------------------------------------------------
        // Render
        //------------------------------------------------

        render: function () {
            var tmpl = TemplateUtil.createTemplate("#asset-list-view-template"),
                noFiles = TemplateUtil.createTemplate("#upload-files-here-template"),
                dragDrop = TemplateUtil.createTemplate("#drag-drop-file-capture-template");

            if (!this.$el[0]) {
                this.setElement("#results");
            }
            this.$el.html(tmpl);
            this.$el.append(noFiles);
            this.$el.append(dragDrop);

            if (this.loading === false) {
                $("#loadingSpinner").hide();
                this.showUploadGuidance();
            }

            if (this.scErrorText !== undefined) {
                $("#errorInfo").html("<h1>Error Accessing Shared Cloud</h1><p>" + this.scErrorText);
            }

            if (this.collection.models.length === 1 && this.collection.models[0].id === undefined) {
                return this;
            }

            var that = this;
            _.each(this.collection.models, function (assetModel) {
                if (assetModel.attributes.type === "application/x-photoshop" ||
                        assetModel.attributes.type === "image/vnd.adobe.photoshop" ||
                        assetModel.attributes.type === "image/x-photoshop" ||
                        assetModel.attributes.type === "image/photoshop" ||
                        assetModel.attributes.type === "application/psd" ||
                        assetModel.attributes.type === "application/photoshop" ||
                        assetModel.attributes.type === "application/vnd.adobe.directory+json") {
                    that.renderAsset(assetModel);
                }
            }, this);

            var dropzone = $('#dropzone');

            if (typeof (Modernizr) !== "undefined" && Modernizr.draganddrop) {
                var $dropTarget = $('#drag_drop_file_capture'),
                    $uploadFiles = $("#upload_files_here");
                that.$el[0].ondragover = function (e) {
                    if (!$.contains(that.$el[0], $dropTarget) && $("#draganddropmessage").is(':visible')) {
                        $dropTarget.show();
                        $("#upload_files_here").hide();
                    }
                    return false;
                };

                $dropTarget[0].ondragleave = function (e) {
                    if ($.contains(that.$el[0], $dropTarget[0])) {
                        $dropTarget.hide();
                        that.showUploadGuidance();
                    }
                };
                $dropTarget[0].ondragend = function (e) {
                    if ($.contains(that.$el[0], $dropTarget[0])) {
                        $dropTarget.hide();
                        that.showUploadGuidance();
                    }
                };
                $dropTarget[0].ondrop = function (e) {
                    if ($.contains(that.$el[0], $dropTarget[0])) {
                        $dropTarget.hide();
                        that.showUploadGuidance();
                    }
                    e.preventDefault();

                    var files = e.dataTransfer.files,
                        filesToUpload = [],
                        i,
                        headersLoaded = 0;

                    _.each(files, function (file) {
                        var blob = file.slice(0, 4, 'UTF-8');
                        var reader = new FileReader();
                        var mimetype = file.type;
                        var currFile = file;
                        var ext = file.name.substr((~-file.name.lastIndexOf(".") >>> 0) + 2).toLowerCase();

                        reader.addEventListener("loadend", function () {
                            if (this.result === "8BPS") { // http://www.adobe.com/devnet-apps/photoshop/fileformatashtml/#50577409_pgfId-1055726
                                filesToUpload.push(currFile);
                            }

                            headersLoaded = headersLoaded + 1;
                            if (headersLoaded === files.length) {
                                if (filesToUpload.length > 0) {
                                    graphite.events.trigger("upload-init", { files: filesToUpload });
                                }
                                if (filesToUpload.length !== files.length) {
                                    graphite.events.trigger('load-alert', deps.translate("CAN_ONLY_UPLOAD_PSD_FILES_COUNT_FILES_NOT_UPLOADED", (files.length - filesToUpload.length), files.length));
                                    graphite.events.trigger('bad-file-type-dragdrop', {extension: ext, type: mimetype});
                                }
                            }
                        });
                        reader.readAsText(blob, 'UTF-8');
                    });

                    return false;
                };
            } else {
                dropzone.hide();
            }
            return this;
        },

        renderAsset: function (item) {
            var view = new AssetView({
                model: item
            });
            
            
            // Calculate bounds and lay out cards dynamically
            var cardMinWidth = 168,
                cardWidth = 200,
                containerWidth = $('#card-holder')[0].getBoundingClientRect().width,
                fittingNumber = Math.floor(containerWidth / cardWidth),
                margin = Math.round((containerWidth - (fittingNumber * cardMinWidth)) / (fittingNumber + 1));
            view.$el.css('margin-left', margin + 'px');
            $("#card-holder").append(view.el);
            this.childViews.push(view);
            $("#upload_files_here").hide();
        },

        renderNewAsset: function (item) {
            //metadata is empty on new uploads, so we need to skip mimetype checks on those
            var mimeType = item ? item.attributes.type : null,
                view;

            if (!mimeType ||
                    mimeType === "application/x-photoshop" ||
                    mimeType === "image/vnd.adobe.photoshop" ||
                    mimeType === "image/x-photoshop" ||
                    mimeType === "image/photoshop" ||
                    mimeType === "application/psd" ||
                    mimeType === "application/photoshop" ||
                    mimeType === "application/vnd.adobe.directory+json") {

                view = new AssetView({
                    model: item
                });
                
				$("#gridContainer").prepend(view.render().el);
                this.childViews.push(view);
            }
        },

        remove: function () {
            
            this.removeEventListeners();
            this.unbind();
            
            this.childViews.forEach(function (view) {
                // Remove view from DOM
                view.remove();
                view.model.destroy();
				view = null;
            });
            
            
            $("#card-holder").empty();
            
            this.childViews.length = 0;
            this.collection.reset();
            this.collection.length = 0;
            
            this.$el.empty();
            this.stopListening();
            return this;
        },

        showUploadGuidance: function () {
            if ($("#gridContainer .card-asset").length === 0) {
                $("#upload_files_here").show();
            } else {
                $("#upload_files_here").hide();
            }
        },

        //------------------------------------------------
        // Upload
        //------------------------------------------------

        uploadFiles: function (data) {
            if (AuthController.isLoggedIn()) {
                var files = data.files,
                    i;
                for (i = 0; i < files.length; i++) {
                    var item = UploadController.upload(files[i]);

                    this.collection.add(item);
                    $("#upload_files_here").hide();
                }
            } else {
                // DPO - I can't work out when or how this method gets fired.
                // There was an alert here before. This would never get shown, because it's not an alert.
                graphite.events.trigger('load-alert', deps.translate("NO_LONGER_LOGGED_IN_LOGIN_AGAIN"));
                window.location.href = window.location;
            }
        }
    });

    return AssetListView;
});
