//////////////////////////////////////////////////////////////////////////////////////////////
//
// ADOBE CONFIDENTIAL
// ___________________
//
//  Copyright 2014 Adobe Systems Incorporated
//  All Rights Reserved.
//
// NOTICE:  All information contained herein is, and remains
// the property of Adobe Systems Incorporated and its suppliers,
// if any.  The intellectual and technical concepts contained
// herein are proprietary to Adobe Systems Incorporated and its
// suppliers and are protected by trade secret or copyright law.
// Dissemination of this information or reproduction of this material
// is strictly forbidden unless prior written permission is obtained
// from Adobe Systems Incorporated.
//////////////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////////////////////
// Summary Comment
// These are the Bridge Functions which are made as utility for PSD Extract Service
//////////////////////////////////////////////////////////////////////////////////////////////

/*jslint plusplus: true */
/*global alert: false, console: false, window: false, document: false, dw: false, dwscripts:false, site:false, DWfile: true */

var DWPSDExtractBridge = {};

//////////////////////////////////////////////////////////////////////////////////////////////
// DW API Utility Methods
//

//--------------------------------------------------------------------
// FUNCTION:
//  applyCallback
//
// DESCRIPTION:
//  Call the DW Browser back with the callback if required
//
// ARGUMENTS:
//  intial argsObj and the result
//
// RETURNS:
//  Nothing
//--------------------------------------------------------------------
DWPSDExtractBridge.applyCallback = function (argsObj, parameter) {
	"use strict";
    if (typeof argsObj.callback === "function") {
        if (parameter !== undefined && parameter !== null) {
            argsObj.callback(parameter);
        } else {
            argsObj.callback();
        }
    }
};

//--------------------------------------------------------------------
// FUNCTION:
//   logArguments
//
// DESCRIPTION:
//  Log the incoming argument for debugging purpose
//
// ARGUMENTS:
//  methodName - method being invoked
//  argObj - arguments passed through JS Bridging
//
// RETURNS:
//  Nothing
//--------------------------------------------------------------------
DWPSDExtractBridge.logArguments = function (methodName, object) {
    "use strict";
    dw.logDebugMsg(methodName + " called in DWPSDExtractBridge");
    var property;
    for (property in object) {
		if (object.hasOwnProperty(property)) {
			dw.logDebugMsg(methodName + ": " + property + ": " + object[property].toString());
		}
    }
    return;
};

//////////////////////////////////////////////////////////////////////////////////////////////
// DW API Bridging Methods
//

//--------------------------------------------------------------------
// FUNCTION:
//   updateCSSForSelection
//
// DESCRIPTION:
//  Call the PSDExtract Object to update the Extracted CSS in Dw
//
// ARGUMENTS:
//   argsObj.cssString :- Proper CSS in string format
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DWPSDExtractBridge.updateCSSForSelection = function (argsObj) {
    "use strict";
    if (argsObj.imageData.url) {
        dw.logDebugMsg(argsObj.imageData.url);
    }
    if (argsObj.imageData.color) {
        dw.logDebugMsg(argsObj.imageData.color.opacity + " " +  argsObj.imageData.color.red + " " + argsObj.imageData.color.green + " " + argsObj.imageData.color.blue);
    }
    dw.logDebugMsg(argsObj.codeHintInfo.cssString + " " + argsObj.imageData.bounds.bottom + " " + argsObj.imageData.bounds.top + " " + argsObj.imageData.bounds.left + " " + argsObj.imageData.bounds.right + " " + argsObj.imageData.isImgAvailable);
    // log the arguments
    DWPSDExtractBridge.logArguments("updateCSSForSelection", argsObj);

    // update the extract css in dw for codehints
    dw.psdExtract.updateCSSForSelection(argsObj.codeHintInfo.cssString, argsObj.codeHintInfo.imagePreviewName, argsObj.imageData.url, argsObj.imageData.bounds.bottom, argsObj.imageData.bounds.top, argsObj.imageData.bounds.left, argsObj.imageData.bounds.right, argsObj.imageData.isImgAvailable, argsObj.imageData.color.opacity, argsObj.imageData.color.red, argsObj.imageData.color.green, argsObj.imageData.color.blue);

    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj);
};

//////////////////////////////////////////////////////////////////////////////////////////////
// DW API Bridging Methods
//

//--------------------------------------------------------------------
// FUNCTION:
//   logExtractEvent
//
// DESCRIPTION:
//  Call the PSDExtract Object to log Extract event in Dw
//
// ARGUMENTS:
//   argsObj.eventname :- eventname for log
//	 argsObj.subevent  :- subevent for log
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DWPSDExtractBridge.logExtractEvent = function (argsObj) {
    "use strict";
    dw.logDebugMsg(argsObj.eventname + argsObj.subevent);
    // log the arguments
    DWPSDExtractBridge.logArguments("logExtractEvent", argsObj);

    //log the event
    if (argsObj.dataGroup) {
        DWPSDExtractBridge.logDataGroupEvent(argsObj);
    } else {
        dw.logEvent(argsObj.eventname, argsObj.subevent);
    }
};

//--------------------------------------------------------------------
// FUNCTION:
//   logDataGroupEvent
//
// DESCRIPTION:
//  Log a data group event 
//
// ARGUMENTS:
//   argsObj.eventname :- eventname for log
//	 argsObj.subevent  :- subevent for log
//	 argsObj.dataGroup :- array of strings where each string is of form "key"+":"+"value"
//
//--------------------------------------------------------------------
DWPSDExtractBridge.logDataGroupEvent = function (argsObj) {
    "use strict";
    
    //while passing between browser and the dw bridging layer, 
    //array has been modified and ahve indexes are changed to string
    var i, valArray = [];
    for (i = 0; i < argsObj.dataGroup.length; ++i) {
        valArray.push(argsObj.dataGroup[String(i)]);
    }
    
    dw.logDataGroupEvent("", argsObj.eventname, argsObj.subevent, valArray);
};

//--------------------------------------------------------------------
// FUNCTION:
//   copyText
//
// DESCRIPTION:
//  Call the PSDExtract Object to Copy the text to the clipboard
//
// ARGUMENTS:
//   argsObj.text :- Proper text
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DWPSDExtractBridge.copyText = function (argsObj) {
	"use strict";
    // log the arguments
	dw.logDebugMsg(argsObj.text);
	DWPSDExtractBridge.logArguments("copyText", argsObj);

    // copy the text to the clipboard
    dw.psdExtract.copyText(argsObj.text);

    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   getClipboardText
//
// DESCRIPTION:
//  Call the PSDExtract Object to return the copied data from the clipboard
//
// ARGUMENTS:
//   argsObj.callback :- Callback to be called with the clipboard data
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DWPSDExtractBridge.getClipboardText = function (argsObj) {
	"use strict";
	DWPSDExtractBridge.logArguments("getClipboardText", argsObj);

    // copy the text to the clipboard
    var clipBoardText = dw.getClipboardText(true);

    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj, clipBoardText);
};

//--------------------------------------------------------------------
// FUNCTION:
//   copyCSS
//
// DESCRIPTION:
//  Call the PSDExtract Object to Copy the CSS to the clipboard
//  and copy dw css flavour for internal use
//
// ARGUMENTS:
//   argsObj.cssString :- Proper CSS in string format
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DWPSDExtractBridge.copyCSS = function (argsObj) {
	"use strict";
    // log the arguments
    DWPSDExtractBridge.logArguments("copyCSS", argsObj);

    // copy the css to the clipboard and also copy a Dw flavor for internal use
    dw.psdExtract.copyCSS(argsObj.cssString, argsObj.gradientString, argsObj.gradientProperty, argsObj.gradientValue, argsObj.gradientVendorString);

    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   pathExists
//
// DESCRIPTION:
//  Verify if the specified file exists in the users system
//
// ARGUMENTS:
//   argsObj.path :- Proper Path to be checked
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DWPSDExtractBridge.pathExists = function (argsObj) {
	"use strict";
    // log the arguments
    DWPSDExtractBridge.logArguments("pathExists", argsObj);

    // check if file exists
    var result = dwscripts.fileExists(argsObj.path);

    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj, result);
};

//--------------------------------------------------------------------
// FUNCTION:
//   browseForFile
//
// DESCRIPTION:
//  Shows the browse for File dialog and returns the selected file
//
// ARGUMENTS:
// Object containing callback
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DWPSDExtractBridge.browseForFile = function (argsObj) {
	"use strict";
    // log the arguments    
    DWPSDExtractBridge.logArguments("browseForFile", argsObj);
    var path = dw.browseForFileURL();

    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj, path);
};

//--------------------------------------------------------------------
// FUNCTION:
//   browseForFolder
//
// DESCRIPTION:
//  Shows the browse for Folder dialog and returns the selected folder path
//
// ARGUMENTS:
// Object containing callback
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DWPSDExtractBridge.browseForFolder = function (argsObj) {
	"use strict";
    var path = '';
    // log the arguments
    DWPSDExtractBridge.logArguments("browseForFolder", argsObj);
    
    if (argsObj.initialDirectory) {
        path = dw.browseForFolderURL(argsObj.title, argsObj.initialDirectory, false, false);
    } else {
        path = dw.browseForFolderURL();
    }
    dw.psdExtract.setFocusToExtractPanel();
    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj, path);
};

DWPSDExtractBridge.refreshContextMenu = function (argsObj) {
    "use strict";
    // log the arguments
    DWPSDExtractBridge.logArguments("refreshContextMenu", argsObj);
    
    dw.psdExtract.refreshContextMenu();
    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   getImageDownloadLocation
//
// DESCRIPTION:
//  Gets the default image download folder for site.
//  If no folder is specified return site root
//
// ARGUMENTS:
// Object containing callback
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------

DWPSDExtractBridge.getImageDownloadLocation = function (argsObj) {
    "use strict";
    DWPSDExtractBridge.logArguments("getImageDownloadLocation", argsObj);
    
    var siteName = site.getCurrentSite(),
        siterootPath = '',
        siterootUrl = '';
    
    if (site) {
        siterootUrl = site.getDefaultImagesFolder();
        if (!siterootUrl) {
            siterootUrl = site.getLocalRootURL(siteName);
            if (!siterootUrl) {
                siterootUrl = argsObj.defaultFolder;
            }
        }
    }
    
    if (!siterootUrl) {
        return;
    }
    
    if (argsObj.returnPath) {
        siterootPath = dw.psdExtract.UrlToPath(siterootUrl);
    }

    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj, {
        siterootPath: siterootPath,
        siterootUrl: siterootUrl
    });
        
};

//--------------------------------------------------------------------
// FUNCTION:
//   deleteFile
//
// DESCRIPTION:
//  Deletes the image at given path
//
// ARGUMENTS:
// Object containing image path and callback
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------

DWPSDExtractBridge.deleteFile = function (argsObj) {
    "use strict";
    DWPSDExtractBridge.logArguments("deleteFile", argsObj);
    
    var result = dw.psdExtract.DeleteFile(argsObj.path);
    
    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj, result);
};

//--------------------------------------------------------------------
// FUNCTION:
//   ValidatePathForExtractAssetDownload
//
// DESCRIPTION:
//  This method is to validate the download path of the asset from Extract Panel
//
// ARGUMENTS:
// string - Path to be validated
//
// RETURNS:
//  Error/Valid code
//--------------------------------------------------------------------

DWPSDExtractBridge.ValidatePathForExtractAssetDownload = function (argsObj) {
    "use strict";
    DWPSDExtractBridge.logArguments("ValidatePathForExtractAssetDownload", argsObj);
    
    var responseCode = dw.psdExtract.ValidatePathForExtractAssetDownload(argsObj.filePath, argsObj.isBatchExport);
    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj, responseCode);
};

//--------------------------------------------------------------------
// FUNCTION:
//   getFilePathsForMultipleAssetsDownload
//
// DESCRIPTION:
//  This method is to to get the file paths based on preferences 
//  for downloading multiple assets
//
// ARGUMENTS:
// string - Path specified by user
//
// RETURNS:
//  Array of Objects
//      Validation Success/Failure
//      Device Type Scale Value
//      Path to file
//--------------------------------------------------------------------

DWPSDExtractBridge.getFilePathsForMultipleAssetsDownload = function (argsObj) {
    "use strict";
    DWPSDExtractBridge.logArguments("getFilePathsForMultipleAssetsDownload", argsObj);
    
    var listSelectedDevicePaths = dw.psdExtract.getFilePathsForMultipleAssetsDownload(argsObj.filePath);
    
    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj, listSelectedDevicePaths);
};

//--------------------------------------------------------------------
// FUNCTION:
//   displayExtractPrefs
//
// DESCRIPTION:
//  This method is to display the settings for extract in preferences dialog
//
//--------------------------------------------------------------------

DWPSDExtractBridge.displayExtractPrefs = function () {
    "use strict";
    dw.showPreferencesDialog('Extract');
};

//--------------------------------------------------------------------
// FUNCTION:
//   writeBase64
//
// DESCRIPTION:
//  Write base64 data to file specifed. File supported - png, jpg, jpeg
//
// ARGUMENTS:
//  argsObj.path :- Proper Path for the file
//  argsObj.contents :- Base 64 encoded data
//  argsObj.accessToken :- Valid DW Access Token to gain access to file system for writing
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DWPSDExtractBridge.writeBase64 = function (argsObj) {
	"use strict";
    // log the arguments
    DWPSDExtractBridge.logArguments("writeBase64", argsObj);

    // attempt to write the base 64 data to a file
    var result = dw.writeBase64(argsObj.path, argsObj.contents, argsObj.accessToken);

    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj, result);
};

var ExtractPrefs = {
    SECTION: "Extract",
    OPTIMISE: "Extract optimise",
    BASE_FONT_SIZE: "Extract Base Font Size",
    IMAGE_FORMAT_EXTRACT: "Extract Image Format",
    PREFERRED_FONT_UNIT: "Extract Preferred Font Unit"
};
//--------------------------------------------------------------------
// FUNCTION:
//   getExtractPreferences
//
// DESCRIPTION:
//  Gets Extract specific preferences
//
// ARGUMENTS:
//  argsObj.callback :- Callback function to be called
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DWPSDExtractBridge.getExtractPreferences = function (argsObj) {
	"use strict";
    // log the arguments
    DWPSDExtractBridge.logArguments("getExtractPreferences", argsObj);
    
    DWPSDExtractBridge.applyCallback(argsObj, {
        preferredImageQuality: dw.getPreferenceInt(ExtractPrefs.SECTION, ExtractPrefs.OPTIMISE, 80),
        baseFontSizeValue: dw.getPreferenceString(ExtractPrefs.SECTION, ExtractPrefs.BASE_FONT_SIZE, "16"),
        preferredFileExtension: dw.getPreferenceString(ExtractPrefs.SECTION, ExtractPrefs.IMAGE_FORMAT_EXTRACT, "png"),
        preferredFontUnit: dw.getPreferenceString(ExtractPrefs.SECTION, ExtractPrefs.PREFERRED_FONT_UNIT, "px")
    });
};

//--------------------------------------------------------------------
// FUNCTION:
//   writeString
//
// DESCRIPTION:
//  Write string data to file specifed.
//
// ARGUMENTS:
//  argsObj.path :- Proper Path for the file
//  argsObj.contents :- File content
//
// RETURNS:
//  Calls the callback if present
//--------------------------------------------------------------------
DWPSDExtractBridge.writeString = function (argsObj) {
	"use strict";
    // log the arguments
    DWPSDExtractBridge.logArguments("writeString", argsObj);

    // attempt to write the content to a file
    var result = DWfile.write(argsObj.path, argsObj.contents);

    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj, result);
};

//--------------------------------------------------------------------
// FUNCTION:
//  backgroundAssetExtractComplete
//
// DESCRIPTION:
//  This is called from the webapp
//  to flag an asset extraction complete. This inturn
//  tells Dreamweaver to handle success message.
//
// ARGUMENTS:
//	none
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.backgroundAssetExtractComplete = function (argsObj) {
    "use strict";
    // Got a background asset extract success. Go ahead and
    // and pass this message to DW so it can refresh
    // itself
    dw.psdExtract.backgroundAssetExtractcomplete(argsObj.filepath, argsObj.infotype, argsObj.nonEncodedPath);
};

// Called from the webapp to indicate failure in asset extraction.
DWPSDExtractBridge.backgroundAssetExtractError = function (argsObj) {
    "use strict";
    // Failure in extracting the asset. Fwd this message to Dw.
    dw.psdExtract.backgroundAssetExtractError(argsObj.filepath);
};

//--------------------------------------------------------------------
// FUNCTION:
//   startDrag
//
// DESCRIPTION:
//  Call the Dreamweaver Object to start OLE Drag
//
// ARGUMENTS:
//   argsObj.flavor :- Flavor of the Drag to be started
//   argsObj.layerids :- Array of layerIDs being dragged
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.startDrag = function (argsObj) {
    "use strict";
    dw.logDebugMsg(argsObj.flavor + " " + argsObj.layerids + " " + argsObj.extensionID + " " + argsObj.layerName);
    // log the arguments
    DWPSDExtractBridge.logArguments("startDrag", argsObj);
    
    // update the extract css in dw for codehints
    dw.startDrag(argsObj.flavor, argsObj.layerids, argsObj.extensionID, argsObj.layerName);

    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   refreshPanel
//
// DESCRIPTION:
//  Call the Dreamweaver Object to refresh the panel
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.refreshPanel = function (argsObj) {
    "use strict";
    //Refresh the panel
    dw.psdExtract.refresh();
    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   setFirstLaunchDismissed
//
// DESCRIPTION:
//  Call the Dreamweaver Object to set a preference indicating whether the First Launch has been dismissed
//
// ARGUMENTS:
//   a boolean flag, isDismissed.
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.setWelcomeExpDismissed = function (argsObj) {
    "use strict";
    dw.psdExtract.setWelcomeExpDismissed(argsObj.isDismissed);
    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   setPrefToShowUploadDialogOnReload
//
// DESCRIPTION:
//  Call the Dreamweaver Object to set a preference indicating whether to open upload PSD dialog on reload.
//
// ARGUMENTS:
//   a boolean flag, preference value to show upload dialog or not.
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.setPrefToShowUploadDialogOnReload = function (argsObj) {
    "use strict";
    dw.psdExtract.setPrefToShowUploadDialogOnReload(argsObj.canShow);
    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   extractReadyForInit
//
// DESCRIPTION:
//  Call the Dreamweaver Object to inform that extract manager is now ready
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.extractReadyForInit = function (argsObj) {
    "use strict";
    
    // log the arguments
    DWPSDExtractBridge.logArguments("extractReadyForInit", argsObj);

    dw.psdExtract.initializeExtract();
    // call the callback if present
    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   setImageAvailable
//
// DESCRIPTION:
//  Call the Dreamweaver Object to inform wheater image extraction service
//  is available or not.
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.setImageAvailable = function (argsObj) {
    "use strict";
    dw.psdExtract.setImageAvailable(argsObj.isImgAvailable);
    // call the callback if present
    //DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   setPSDAvailable
//
// DESCRIPTION:
//		This method is to be called to set that a psd image is loaded in extract panel
//
// ARGUMENTS:
//   argsObj.isPSDAvailable if a psd is available in panel
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.setPSDAvailable = function (argsObj) {
    "use strict";
    dw.psdExtract.setPSDAvailable(argsObj.isPSDAvailable);
    // call the callback if present
    //DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   setSelectionBounds
//
// DESCRIPTION:
//		This method is to be called to set The selection bounds of the selected layer
//
// ARGUMENTS:
//   argsObj.left, argsObj.top, argsObj.right, argsObj.bottom of the selection relative to the panel top
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.setSelectionBounds = function (argsObj) {
    "use strict";
    dw.psdExtract.setSelectionBounds(argsObj.left, argsObj.top, argsObj.right, argsObj.bottom);
    // call the callback if present
    //DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   hideWelcomePageOnUpload
//
// DESCRIPTION:
//  Call the Dreamweaver Object to hide the welcome screen.
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.hideWelcomePageOnUpload = function (argsObj) {
    "use strict";
    var isFloating = dw.psdExtract.isExtractPanelFloating(),
		forceHide = false,
		isVisible = dw.isWelcomePageVisible();
	
    /* If panel is docked and OS is MAC, we need to hide the welcome screen. */
    if (!isFloating && isVisible && dw.psdExtract.IsMacOS()) {
        dw.showWelcomePage(false);
		forceHide = true;
    }
    
    DWPSDExtractBridge.applyCallback(argsObj, forceHide);
};

//--------------------------------------------------------------------
// FUNCTION:
//   showWelcomePageIfNeeded
//
// DESCRIPTION:
//  Call the Dreamweaver Object to show the welcome screen.
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.showWelcomePageIfNeeded = function (argsObj) {
    "use strict";
    dw.showWelcomePage(true);
    
    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   clickInExtractPanel
//
// DESCRIPTION:
//  Call the Dreamweaver Object to simulate a user click.
//
// ARGUMENTS:
//   none
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.clickInExtractPanel = function (argsObj) {
    "use strict";
    dw.psdExtract.clickInExtractPanel(argsObj.x, argsObj.y);

    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   setOWLBoolUsePersistentDrawer
//
// DESCRIPTION:
//  Use this API to set/unset the OWL theme flag for peristent drawers
//
// ARGUMENTS:
//   a boolean flag to set/unset the OWL theme flag for peristent drawers
//
// RETURNS:
//  none
//--------------------------------------------------------------------
DWPSDExtractBridge.setOWLBoolUsePersistentDrawer = function (argsObj) {
    "use strict";
    dw.psdExtract.setOWLBoolUsePersistentDrawer(argsObj.panelPersistentDrawer);
    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   getOWLBoolUsePersistentDrawer
//
// DESCRIPTION:
//  Use this API to get the OWL theme flag for persistent drawers
//
// ARGUMENTS:
//   None
//
// RETURNS:
//   a boolean flag to get the OWL theme flag for persistent drawers
//--------------------------------------------------------------------
DWPSDExtractBridge.getOWLBoolUsePersistentDrawer = function (argsObj) {
    "use strict";
    var isPersistent = dw.psdExtract.getOWLBoolUsePersistentDrawer();
    DWPSDExtractBridge.applyCallback(argsObj, isPersistent);
};

//--------------------------------------------------------------------
// FUNCTION:
//   openInBrowser
//
// DESCRIPTION:
//  Open the url that is passed in the primary browser
//
// ARGUMENTS:
//   url: url to be opened in the browser
//
// RETURNS:
//   Nothing
//--------------------------------------------------------------------
DWPSDExtractBridge.openInBrowser = function (argsObj) {
    "use strict";
    dw.browseDocument(argsObj.url);
    
    DWPSDExtractBridge.applyCallback(argsObj);
};

//--------------------------------------------------------------------
// FUNCTION:
//   isAnyDocumentOpened
//
// DESCRIPTION:
//  Checks whether there is any document opened.
//
// ARGUMENTS:
//   
//
// RETURNS:
//   A boolean indicating if any document is opened for edit.
//--------------------------------------------------------------------
DWPSDExtractBridge.isAnyDocumentOpened = function (argsObj) {
    "use strict";
    var retArg = (dw.getDocumentDOM()) ? true : false;
    
    DWPSDExtractBridge.applyCallback(argsObj, retArg);
};