/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2016 Adobe Systems Incorporated                          */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE:  All information contained herein is, and remains           */
/* the property of Adobe Systems Incorporated and its suppliers,       */
/* if any.  The intellectual and technical concepts contained          */
/* herein are proprietary to Adobe Systems Incorporated and its        */
/* suppliers and are protected by all applicable intellectual property */
/* laws, including trade secret and copyright laws.                    */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe Systems Incorporated.                                    */
/*                                                                     */
/***********************************************************************/

var gWorkflowTriggerALL = null;

function Workflow(/*Boolean*/ inIsRemote)
{
    this.id = '';
    this.name = '';
	this.date = new Date();
	this.remote = isValidProperty(inIsRemote) ? inIsRemote : false;
    this.content = new WorkflowContent();
    this.steps = [];
    this.stepswitch = 'automatic';
	this.session = '';	// uuid for each workflow execution
    
	//////////////////////////////////////////////////////////////////////////////
	//
	// Initialize with JSON (wfd V1)
	//
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
	    throwInvalid(inJSON);

		// workflow description version
		var jsonVersion_2 = inJSON['v2'];
		if (isValidProperty(jsonVersion_2))
		{
			this.initializeJSON_V2(jsonVersion_2)
		}
		else
		{
			this.id = inJSON['id'];
			throwInvalid(this.id);

			var name = inJSON['name'];
			if (isValidProperty(name))
			{
				this.name = name;
			}

			var date = inJSON['date'];
			if (isValidProperty(date) && !isNaN(Date.parse(date)))
			{
				this.date = new Date(date);
			}

			this.stepswitch = inJSON['stepswitch'];
			if (isValidProperty(this.stepswitch))
			{
				if (this.stepswitch != Workflow.STEPSWITCH_AUTOMATIC &&
					this.stepswitch != Workflow.STEPSWITCH_MANUALLY)
				{
					this.stepswitch = Workflow.STEPSWITCH_AUTOMATIC;
				}
			}
			else
			{
				this.stepswitch = Workflow.STEPSWITCH_AUTOMATIC;
			}

			// add content to the workflow
			if (inJSON.hasOwnProperty('content'))
			{
				this.content.initializeJSON(inJSON['content']);
			}
			else
			{
				throwlog("Missing Workflow Content");
			}

			// add steps
			var stepsArray = inJSON['steps'];
			throwInvalid(stepsArray, 'Array');

			// make this.steps visible to forEach-callback function
			var steps = this.steps;
			var thisObj = this;

			forEach(stepsArray, function (stepObj)
			{
				var step = new WorkflowStep(thisObj);
				step.initializeJSON(stepObj);
				steps.push(step);
			});
		}
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Initialize with JSON (wfd V2)
	//
	this.initializeJSON_V2 = function(/*[Object]*/ inJSON)
	{
		throwInvalid(inJSON);

		var appVersion = inJSON['version'];
		var platform = inJSON['platform'];

		if (validateSystem(appVersion, platform))
		{
			this.id = inJSON['id'];
			throwInvalid(this.id);

			var name = inJSON['name'];
			if (isValidProperty(name))
			{
				this.name = name;
			}

			var date = inJSON['date'];
			if (isValidProperty(date) && !isNaN(Date.parse(date)))
			{
				this.date = new Date(date);
			}

			this.stepswitch = inJSON['stepswitch'];
			if (isValidProperty(this.stepswitch))
			{
				if (this.stepswitch != Workflow.STEPSWITCH_AUTOMATIC &&
					this.stepswitch != Workflow.STEPSWITCH_MANUALLY)
				{
					this.stepswitch = Workflow.STEPSWITCH_AUTOMATIC;
				}
			}
			else
			{
				this.stepswitch = Workflow.STEPSWITCH_AUTOMATIC;
			}

			// add trigger components to the workflow
			if (inJSON.hasOwnProperty('components'))
			{
				try
				{
					WorkflowTriggerComponentManager.get().initializeWorkflowJSON(this, inJSON['components']);
				}
				catch(exc)
				{
					exclog(exc);
				}
			}

			// add content to the workflow
			if (inJSON.hasOwnProperty('content'))
			{
				this.content.initializeJSON(inJSON['content']);
			}
			else
			{
				throwlog("Missing Workflow Content");
			}

			// add steps
			var stepsArray = inJSON['steps'];
			throwInvalid(stepsArray, 'Array');

			// make this.steps visible to forEach-callback function
			var steps = this.steps;
			var thisObj = this;

			forEach(stepsArray, function (stepObj)
			{
				var step = new WorkflowStep(thisObj);
				step.initializeJSON(stepObj);
				steps.push(step);
			});
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get content for locator
	//
    this.getContent = function(/*[string]*/ inLocator)
    {
	    throwInvalid(this.id);
    	return this.content.getContent(inLocator);
    }
     
	//////////////////////////////////////////////////////////////////////////////
	//
	// Get all known content locator IDs
	//
    this.getContentLocator = function()
    {
	    throwInvalid(this.id);
    	return this.content.getContentLocator();
    }
     
	//////////////////////////////////////////////////////////////////////////////
	//
	// Get WorkflowStep for given index
	//
    this.getStep = function(/*[Number]*/ inStepNumber)
    {
	    throwInvalid(this.id);
    	var ret = null;
    	
    	if (inStepNumber >= 0 && inStepNumber < this.steps.length)
    	{
    		ret = this.steps[inStepNumber];
    	}
    	
    	return ret;
    }
    
	//////////////////////////////////////////////////////////////////////////////
	//
	// Get number of WorkflowStep's
	//
    this.getStepLength = function()
    {
    	return this.steps.length;
    }
}

Workflow.STEPSWITCH_AUTOMATIC		= 'automatic';
Workflow.STEPSWITCH_MANUALLY		= 'manually';
Workflow.STEPSWITCH_STEP_AUTOMATIC	= 'step_automatic';

//////////////////////////////////////////////////////////////////////////////
 
function WorkflowStep(/*[Workflow]*/ inWorkflow, /*WorkflowStep*/ inParentStep)
{
    this.indicator = [];
    this.exe = null;
    this.triggers = [];
    this.content = new WorkflowContent();
	this.preConditions = [];
	this.stepswitch = inWorkflow.stepswitch;
	this.switchtimeout = NaN;

	this.parentStep = isValidProperty(inParentStep) ? inParentStep : null;
	this.subSteps = [];
	this.subStepOrder = "sequence";
    
    var workflow = inWorkflow;
    
	//////////////////////////////////////////////////////////////////////////////
	//
	// Initialize with JSON
	//
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
	    throwInvalid(inJSON);

		// read indicators (optional)
		//
		var indicator = inJSON['indicator'];

		if (isValidProperty(indicator))
		{
			if (isValidProperty(indicator.length))
			{
				var indicators = this.indicator;
				var thisObj = this;

				forEach(indicator, function (aIndicator)
				{
					try
					{
						var indicatorObj = new Indicator(thisObj);
						indicatorObj.initializeJSON(aIndicator);
						indicators.push(indicatorObj);
					}
					catch (exc)
					{
						exclog(exc);
					}
				});
			}
			else
			{
				try
				{
					var indicatorObj = new Indicator(this);
					indicatorObj.initializeJSON(indicator);
					this.indicator.push(indicatorObj);
				}
				catch (exc)
				{
					exclog(exc);
				}
			}
		}

		// read content (needs to happen before reading sub-steps,
		// since sub-steps inherit the content of their parent)
		//
		var contentArray = inJSON['content'];

		if (isValidProperty(contentArray))
		{
			this.content.initializeJSON(contentArray);
		}
		else
		{
			if (this.isSubStep())
			{
				// sub-steps could inherit content of its parent step
				this.content = this.parentStep.content;
			}
			else
			{
				throwInvalid(contentArray);
			}
		}

		// if this is a sub-step then merge parent content into this sub-step
		//
		if (this.isSubStep())
		{
			this.content.append(this.parentStep.content);
		}

		// read sub-steps (optional)
		//
		var subSteps = inJSON['substeps'];

		if (isValidProperty(subSteps))
		{
			this.initializeSubStepJSON(subSteps);
		}

		// read trigger (skip if sub-steps are defined)
		//
		if (!this.hasSubSteps())
		{
			// add trigger events to the sep
			var triggerArray = inJSON['trigger'];

			if (isValidProperty(triggerArray))
			{
				throwInvalid(triggerArray, 'Array');

				// make this.triggers visible for forEach-callback function
				var triggers = this.triggers;

				forEach(triggerArray, function (triggerObj)
				{
					var trigger = new WorkflowTrigger();
					trigger.initializeJSON(triggerObj);
					triggers.push(trigger);
				});
			}
			else if (this.isSubStep())
			{
				// trigger definitions are mandatory for sub-steps
				throwInvalid(triggerArray);
			}
		}

		if (this.isSubStep() && triggers.length == 0)
		{
			// sub steps must have trigger
			throwlog("WorkflowStep: Sub steps must have trigger definitions.");
		}

		// read toggle method to next step (optional)
		//
		var toggle = inJSON['toggle'];

		if (isValidProperty(toggle) && typeof(toggle) == 'object')
		{
			if (toggle.hasOwnProperty('method'))
			{
				switch (toggle.method)
				{
					case Workflow.STEPSWITCH_AUTOMATIC:
					case Workflow.STEPSWITCH_MANUALLY:
					case Workflow.STEPSWITCH_STEP_AUTOMATIC:
						this.stepswitch = toggle.method;
						break;
					default:
						dbglogInfo("WorkflowStep: Invalid toggle method.");
				}
			}

			if (toggle.hasOwnProperty('timeout'))
			{
				var timeout = parseInt(toggle.timeout);

				if (!isNaN(timeout))
				{
					this.switchtimeout = timeout;
				}
			}
		}

		// read preConditions (optional)
		//
		if (isValidProperty(inJSON['preConditions']))
		{
			throwInvalid(inJSON['preConditions'], 'Array');
			this.preConditions = inJSON['preConditions'];
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Initialize sub-steps with JSON
	//
	this.initializeSubStepJSON = function(/*[Object]*/ inJSON)
	{
		throwInvalid(inJSON);
		throwInvalid(inJSON['steps'], 'Array');

		if (isValidProperty(inJSON['order']))
		{
			this.subStepOrder = (inJSON['order'] == 'ordered' ? 'ordered' : 'any');
		}

		// make this.steps visible to forEach-callback function
		var stepsArray = inJSON['steps'];
		var steps = this.subSteps;
		var thisObj = this;

		forEach(stepsArray, function (stepObj)
		{
			var step = new WorkflowStep(workflow, thisObj);
			step.initializeJSON(stepObj);
			steps.push(step);
		});
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Evaluate precondition scripts
	//
	this.validatePreConditions = function(/*Function*/ inSuccessFct, /*Function*/ inFailureFct)
	{
		var wfstep = this;

		if (isValidProperty(wfstep) && wfstep.preConditions.length > 0)
		{
			var preconditions = wfstep.preConditions;
			var precondIndex = 0;
			var precondResult = true;

			var successFct = inSuccessFct;
			var failureFct = inFailureFct;

			function evalCondition(/*Number*/ inIndex)
			{
				var script = 'validateCondition("' + preconditions[precondIndex] + '", "", ' + (__loglevel__ > DebugLog.LOGLEVEL_NONE) + ');';

				dbglogInfo('WorkflowStep preCondition - Workflow: "' + workflow.name + '" [' + workflow.id + '] PRE-CONDITION-SCRIPT "' + script + '"');

				var job = new ScriptJob(script);
				job.wait(function(/*[Boolean]*/ inResult)
					{
						dbglogError('WorkflowStep preCondition - Workflow: "' + workflow.name + '" [' + workflow.id + '] RESULT: "' + !inResult + '"');
						onConditionEvaluated(!(isValidProperty(inResult) && inResult == false));
					},
					function(/*[String]*/ inError)
					{
						dbglogError('WorkflowProcessor preCondition - Workflow: "' + workflow.name + '" [' + workflow.id + '] ERROR: "' + inError + '"');
						onConditionEvaluated(true);
					});
			}

			function onConditionEvaluated(/*Boolean*/ inResult)
			{
				precondResult &= inResult;
				precondIndex++;

				if (precondIndex < preconditions.length)
				{
					evalCondition(precondIndex);
				}
				else
				{
					if (precondResult)
					{
						successFct();
					}
					else
					{
						failureFct();
					}
				}
			}

			evalCondition(precondIndex);
		}
		else
		{
			inSuccessFct();
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get content for locator ID
	//
    this.getContent = function(/*[string]*/ inLocator)
    {
    	return this.content.getContent(inLocator);
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get indicator
	//
    this.getIndicator = function(/*[Number]*/ inIndex)
    {
    	return this.indicator[inIndex];
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get all indicator
	//
    this.getAllIndicator = function()
    {
    	return this.indicator;
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get all known locator IDs
	//
    this.getContentLocator = function()
    {
    	return this.content.getContentLocator();
    }
    
	//////////////////////////////////////////////////////////////////////////////
	//
	// Return owning worklfow
	//
    this.getWorkflow = function()
    {
    	return workflow;
    }
    
	//////////////////////////////////////////////////////////////////////////////
	//
	// Get all WorkflowTrigger
	//
    this.getTriggers = function()
    {
    	var ret = [];
    	
    	forEach(this.triggers, function(trigger)
    	{
    		var component = trigger.getComponent(workflow);
    		
    		if (isValidProperty(component))
    		{
    			ret = ret.concat(component.getTrigger(trigger));
    		}
    		else
    		{
    			ret.push(trigger);
    		}
    	});

    	return ret;
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Am I a sub-step
	//
	this.isSubStep = function()
	{
		return isValidProperty(this.parentStep);
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Has sub-steps
	//
	this.hasSubSteps = function()
	{
		return (this.subSteps.length > 0);
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get WorkflowStep for given index
	//
	this.getSubStep = function(/*[Number]*/ inSubStepNumber)
	{
		if (!this.hasSubSteps())
		{
			throwlog("Missing sub-steps");
		}

		var ret = null;

		if (inSubStepNumber >= 0 && inSubStepNumber < this.subSteps.length)
		{
			ret = this.subSteps[inSubStepNumber];
		}

		return ret;
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get number of WorkflowStep's
	//
	this.getSubStepLength = function()
	{
		return this.subSteps.length;
	}
}

//////////////////////////////////////////////////////////////////////////////

function Indicator(/*[WorkflowStep]*/ inStep)
{
	this.menu = null;
	this.ctrl = null;
	this.text = null;
	this.direction = null;
	
	var step = inStep;
	var displaying = false;

	//////////////////////////////////////////////////////////////////////////////
	//
	// Initialize with JSON
	//
    this.initializeJSON = function(/*[Object]*/ inJSON)
    {
	    throwInvalid(inJSON);

		var menu = inJSON['menu'];
		var ctrl = inJSON['ctrl'];
		var text = inJSON['text'];
		var direction = inJSON['direction'];
		
		if (isValidProperty(menu) && menu.length)
		{
			this.menu = menu;
		}
		else
		{
			if (isValidProperty(ctrl) && ctrl.length)
			{
				this.ctrl = ctrl;
				this.text = isValidProperty(text) ? text : '';
				this.direction = isValidProperty(direction) ? direction : 'left';
			}
			else
			{
				throwlog("Empty Indicator definition");
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Is this a menu item indicator?
	//
	this.isMenu = function()
	{
		return isValidProperty(this.menu);
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Is this a ui control indicator?
	//
	this.isControl = function()
	{
		return isValidProperty(this.ctrl);
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Display indicator
	//
	this.display = function(/*[IContentStorage]*/ inContentStorage)
	{
		displaying = true;

		if (this.isMenu())
		{
			var script = 'displayMenuIndicator("' + this.menu + '");';
			var job = new ScriptJob(script);
			job.wait(null, function(/*[String]*/ inError)
			{
				dbglogError(inError);
			});
		}
		else
		{
			var text = this.text;
			var ctrl = this.ctrl;
			var direction = this.direction;

			function doDisplay(/*[String]*/ inText)
			{
				if (displaying)
				{
					var text = inText.replace(/\n/g, "\\n");

					if (isValidProperty(ctrl))
					{
						var script = 'displayUIIndicator("' + ctrl + '", "' + text + '", "' + direction + '");';
						var job = new ScriptJob(script);
						job.wait(null, function(/*[String]*/ inError)
						{
							dbglogError(inError);
						});
					}
				}
			}

			var worklfow = null;
			if (isValidProperty(inContentStorage))
			{
				if (isValidProperty(step))
				{
					workflow = step.getWorkflow();
				}
			}

			if (isValidProperty(workflow))
			{
				var job = inContentStorage.getString(workflow.id, text);
				job.wait(function(/*[String]*/ inContent)
				{
					if (isValidProperty(inContent) && inContent.length > 0)
					{
						text = inContent;
					}

					doDisplay(text);
				});
			}
			else
			{
				doDisplay(text);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Cancel asynchronous display call
	//
	this.cancelDisplay = function()
	{
		displaying = false;
	}
}

//////////////////////////////////////////////////////////////////////////////

function WorkflowTrigger()
{
	this.eventType = null;
	this.timer = false;
	this.condition = null;
	this.ignore = false
	this.component = null;
	this.match = WorkflowTrigger.MATCHTYPE_POSITIVE;
	this.isComponent = false;
	this.content = null;
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Return true if there's content defined for this trigger
	//
	this.hasContent = function()
	{
		return isValidProperty(this.content);
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Is a match a success (or a failure)
	//
	this.isPositiveMatch = function()
	{
		return this.match == WorkflowTrigger.MATCHTYPE_POSITIVE;
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Initialize with JSON
	//
	this.initializeJSON = function(/*[Object]*/ inJSON)
	{
	    throwInvalid(inJSON);
	    
	    var evType = inJSON['event'];
	    var timer = inJSON['timer'];
	    var condition = inJSON['condition'];
	    var ignore = inJSON['ignore'];
	    var component = inJSON['name'];
	    var match = inJSON['match'];
	    
	    if (isValidProperty(evType, 'String'))
	    {
	    	this.eventType = evType;
	    }
	    if (isValidProperty(timer, 'Boolean'))
	    {
	    	this.timer = timer;
	    }
	    if (isValidProperty(condition, 'String'))
	    {
	    	this.condition = condition;
	    }
	    if (isValidProperty(ignore, 'Boolean'))
	    {
	    	this.ignore = ignore;
	    }
	    if (isValidProperty(component, 'String'))
	    {
		    this.component = component;
		}
	    if (isValidProperty(match, 'String') &&
	    	(match == WorkflowTrigger.MATCHTYPE_POSITIVE || match == WorkflowTrigger.MATCHTYPE_NEGATIVE))
	    {
		    this.match = match;
		    
		    if (match == WorkflowTrigger.MATCHTYPE_NEGATIVE)
		    {
		    	this.ignore = false;
		    }
		}
		if (inJSON.hasOwnProperty('content'))
		{
			this.content = new WorkflowContent();
			this.content.initializeJSON(inJSON['content']);
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Return related component if defined
	//
	this.getComponent = function(/*Workflow*/ inWorkflow)
	{
		var ret = null;
		
		if (this.component == 'all')
		{
			ret = getTriggerALL();
		}
		else
		{
			ret = WorkflowTriggerComponentManager.get().getComponent(this.component);

			if (!isValidProperty(ret) && isValidProperty(inWorkflow))
			{
				ret = WorkflowTriggerComponentManager.get().getComponent(inWorkflow.id + this.component);
			}
		}
		
		return ret;
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Validate event for trigger
	//
	this.validate = function(/*[EventObject]*/ inEvent, /*[Function]*/ inResultCallback, /*[Boolean]*/ inIgnoreCondition)
	{
		var ignoreCondition = false;
		if (isValidProperty(inIgnoreCondition))
		{
			ignoreCondition = inIgnoreCondition;
		}
		
		if (this.component == 'all')
		{
			// match to everything
			inResultCallback(this, inEvent, true);
		}
		else
		{
			var ret = false;
			var event = inEvent;
		
			if (this.eventType == event.type ||
				(this.timer && event.type == WorkflowTrigger.EVENTTYPE_TIMER))
			{
				if (!ignoreCondition && isValidProperty(this.condition) && this.condition.length)
				{
					var script = 'validateCondition("' + this.condition + '", "' + inEvent.data + '", ' + (__loglevel__ > DebugLog.LOGLEVEL_NONE) + ');';

					dbglogInfo('WorkflowTrigger ("' + this.eventType + '") CONDITION-SCRIPT "' + script + '"');
					
					var thisObj = this;

					var job = new ScriptJob(script);
					job.wait(function(/*[Boolean]*/ inResult)
					{
						dbglogInfo('WorkflowTrigger ("' + thisObj.eventType + '") CONDITION-RESULT "' + inResult + '"');
						inResultCallback(thisObj, inEvent, inResult);
					},
					function(/*[String]*/ inError)
					{
						dbglogError('WorkflowTrigger ("' + thisObj.eventType + '") CONDITION-RESULT "' + inError + '"');
						inResultCallback(thisObj, inEvent, false);
					});
				}
				else if (!this.timer)
				{
					inResultCallback(this, inEvent, true);
				}
			}
			else
			{
				inResultCallback(this, inEvent, false);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Clone this instance
	//
	this.clone = function()
	{
		var ret = new WorkflowTrigger();
		ret.eventType	= this.eventType;
		ret.timer 		= this.timer; 	
		ret.condition 	= this.condition;
		ret.ignore 		= this.ignore; 
		ret.component 	= this.component;
		ret.isComponent = this.isComponent;
		return ret;
	}
}

WorkflowTrigger.EVENTTYPE_TIMER = '<timer>';
WorkflowTrigger.SYMBOL_EVENT_OBJECT = "${EVOBJ}$";
WorkflowTrigger.MATCHTYPE_POSITIVE = 'positive';
WorkflowTrigger.MATCHTYPE_NEGATIVE = 'negative';

//////////////////////////////////////////////////////////////////////////////

function WorkflowTriggerComponent()
{
	this.name = '';
	this.triggers = [];

	//////////////////////////////////////////////////////////////////////////////
	//
	// Initialize with JSON
	//
	this.initializeJSON = function(/*[Object]*/ inJSON)
    {
		// add content to the step
		this.name = inJSON['name'];
		throwInvalid(name);
		
		// add trigger events to the sep
		var triggerArray = inJSON['trigger'];
		throwInvalid(triggerArray, 'Array');

		// make this.triggers visible for forEach-callback function
		var triggers = this.triggers;
		
		forEach(triggerArray, function(triggerObj)
		{
			var trigger = new WorkflowTrigger();
			trigger.initializeJSON(triggerObj);
			trigger.isComponent = true;
			triggers.push(trigger);
		});
    }
    
	//////////////////////////////////////////////////////////////////////////////
	//
	// Return all trigger
	//
	this.getTrigger = function(/*[WorkflowTrigger]*/ inOwner)
    {
    	var ret = [];
    	
    	if (isValidProperty(inOwner))
    	{
    		forEach(this.triggers, function(trigger)
    		{
    			var itrigger = trigger.clone();

				if (inOwner.ignore)
    			{
					// if the component is used with the ignore flag then
					// set the ignore flag of each trigger of the component
					//
	    			itrigger.ignore = true;
					itrigger.match = WorkflowTrigger.MATCHTYPE_POSITIVE
	    		}
	    		else if (!inOwner.isPositiveMatch() && !itrigger.ignore)
				{
					// if the component is used with a negative match
					// then turn the match of each trigger of the component around
					//
					itrigger.match = (itrigger.match == WorkflowTrigger.MATCHTYPE_POSITIVE ?
										WorkflowTrigger.MATCHTYPE_NEGATIVE :
										WorkflowTrigger.MATCHTYPE_POSITIVE);
				}
	    		
    			ret.push(itrigger);
    		});
    	}
	    
	    return ret;
    }
}

//////////////////////////////////////////////////////////////////////////////

function WorkflowContent()
{
	var parts = {};
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Initialize with JSON
	//
	this.initializeJSON = function(/*[Object]*/ inJSON)
    {
	    throwInvalid(inJSON);
	    
		for (var ci in inJSON)
		{
			var contentPart = inJSON[ci];
			throwInvalid(contentPart);

			if (contentPart.hasOwnProperty('id') && contentPart.hasOwnProperty('mime'))
			{
				parts[ci] = new WorkflowContentPart(contentPart['id'], contentPart['mime']);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Append all parts of inContent which don't exists yet
	//
	this.append = function(/*WorkflowContent*/ inContent)
	{
		var locator = inContent.getContentLocator();

		for (var i=0; i<locator.length; i++)
		{
			if (!isValidProperty(parts[locator[i]]))
			{
				parts[locator[i]] = inContent.getContent(locator[i]);
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get content for locator ID
	//
    this.getContent = function(/*[string]*/ inLocator)
    {
    	return parts[inLocator];
    }

	//////////////////////////////////////////////////////////////////////////////
	//
	// Get all known locator IDs
	//
    this.getContentLocator = function()
    {
    	var ret = [];
    	
    	for (var pi in parts)
    	{
    		ret.push(pi);
    	}
    	
    	return ret;
    }
}
 
//////////////////////////////////////////////////////////////////////////////

function WorkflowContentPart(/*[string]*/ inID, /*[string]*/ inMimeType)
{
    throwInvalid(inID);
    throwInvalid(inMimeType);
     
    this.id = inID;
    this.mimeType = inMimeType;
}
 
//////////////////////////////////////////////////////////////////////////////

Workflow.create = function(/*[String]*/ inJSON, /*Boolean*/ inIsRemote)
{
	var wf = null;
	
	try
	{
		var obj = JSON.parse(inJSON);
		wf = new Workflow(inIsRemote);
		wf.initializeJSON(obj);
	}
	catch(exc)
	{
		exclog(exc);
		wf = null;
	}
		
	return wf;
}

//////////////////////////////////////////////////////////////////////////////

function getTriggerALL()
{
	if (!isValidProperty(gWorkflowTriggerALL))
	{
		gWorkflowTriggerALL = new WorkflowTriggerComponent();
		gWorkflowTriggerALL.name = "all";
		var triggerALL = new WorkflowTrigger();
		triggerALL.component = "all";
		gWorkflowTriggerALL.triggers.push(triggerALL);
	}
	
	return gWorkflowTriggerALL;
}