#!/usr/bin/env bash
set -eu

VERSION_FILE="src/res/version"
if [ ! -f "$VERSION_FILE" ]; then
    echo "ERROR: Version file not found: $VERSION_FILE"
    exit 1
fi
VERSION=$(cat "$VERSION_FILE" | tr -d '\n' | tr -d '\r')

LINUX_BIN_NAME=${LINUX_BIN_NAME:-"ACCELA-${VERSION}-linux-binary.tar.gz"}
LINUX_APP_NAME=${LINUX_APP_NAME:-"ACCELA-${VERSION}-linux-appimage.tar.gz"}
LINUX_SRC_NAME=${LINUX_SRC_NAME:-"ACCELA-${VERSION}-linux-source.tar.gz"}
WINDOWS_BIN_NAME=${WINDOWS_BIN_NAME:-"ACCELA-${VERSION}-windows-binary.zip"}

LINUX_BIN_PATH=${LINUX_BIN_PATH:-"dist/linux_bin"}
LINUX_APP_PATH=${LINUX_APP_PATH:-"dist/linux_appimage"}
LINUX_SRC_PATH=${LINUX_SRC_PATH:-"dist/linux_src"}
WINDOWS_BIN_PATH=${WINDOWS_BIN_PATH:-"dist/windows_bin"}

OUTPUT_DIR="out"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

set +eu
# Check if notify-send exists globally
command -v notify-send &> /dev/null
NOTIFY_SEND_AVAILABLE=$?
set -eu

# Logging functions
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 "INFO" "$1" 2>/dev/null
    fi
    set -eu
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 -u normal "WARNING" "$1" 2>/dev/null
    fi
    set -eu
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 -u critical "ERROR" "$1" 2>/dev/null
    fi
    set -eu
}

main() {
    echo "Packaging ACCELA version $VERSION..."

    # Create output directory
    mkdir -p "$OUTPUT_DIR"

    # Package Linux binaries
    if [ -d "$LINUX_BIN_PATH" ] && [ -n "$(ls -A $LINUX_BIN_PATH 2>/dev/null)" ]; then
        log_info "Creating $LINUX_BIN_NAME"
        rm -f "$OUTPUT_DIR"/ACCELA-*-linux-binary.tar.gz
        tar -czf "$OUTPUT_DIR/$LINUX_BIN_NAME" -C $LINUX_BIN_PATH .
    else
        log_warn "Skipping Linux binaries ($LINUX_BIN_PATH not found or empty)"
    fi

    # Package Linux appimage
    if [ -d "$LINUX_APP_PATH" ] && [ -n "$(ls -A $LINUX_APP_PATH 2>/dev/null)" ]; then
        log_info "Creating $LINUX_APP_NAME"
        rm -f "$OUTPUT_DIR"/ACCELA-*-linux-appimage.tar.gz
        tar -czf "$OUTPUT_DIR/$LINUX_APP_NAME" -C $LINUX_APP_PATH .
    else
        log_warn "Skipping Linux appimage ($LINUX_APP_PATH not found or empty)"
    fi

    # Package Linux source
    if [ -d "$LINUX_SRC_PATH" ] && [ -n "$(ls -A $LINUX_SRC_PATH 2>/dev/null)" ]; then
        log_info "Creating $LINUX_SRC_NAME"
        rm -f "$OUTPUT_DIR"/ACCELA-*-linux-source.tar.gz
        tar -czf "$OUTPUT_DIR/$LINUX_SRC_NAME" -C $LINUX_SRC_PATH .
    else
        log_warn "Skipping Linux source ($LINUX_SRC_PATH not found or empty)"
    fi

    # Package Windows binaries
    if [ -d "$WINDOWS_BIN_PATH" ] && [ -n "$(ls -A $WINDOWS_BIN_PATH 2>/dev/null)" ]; then
        log_info "Creating $WINDOWS_BIN_NAME"
        rm -f "$OUTPUT_DIR"/ACCELA-*-windows-binary.zip
        (cd $WINDOWS_BIN_PATH && zip -qr "../../$OUTPUT_DIR/$WINDOWS_BIN_NAME" .)
    else
        log_warn "Skipping Windows binaries ($WINDOWS_BIN_PATH not found or empty)"
    fi

    # Show what we created
    echo ""
    log_info "Created packages in $OUTPUT_DIR:"
    ls -lh "$OUTPUT_DIR" 2>/dev/null || log_warn "No packages were created"
}

main "$@"
