#!/usr/bin/env bash
set -eu

DEST_DIR="dist/windows_bin"
BUILD_DIR_PARENT="build"
BUILD_DIR="$BUILD_DIR_PARENT/windows_bin"

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

set +eu
# Check if notify-send exists globally
command -v notify-send &> /dev/null
NOTIFY_SEND_AVAILABLE=$?
set -eu

# Logging functions
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 "INFO" "$1" 2>/dev/null
    fi
    set -eu
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 -u normal "WARNING" "$1" 2>/dev/null
    fi
    set -eu
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 -u critical "ERROR" "$1" 2>/dev/null
    fi
    set -eu
}

# Change to the directory where this script is located
SCRIPT_DIR="$(dirname "$(realpath "$0")")"
cd "$SCRIPT_DIR"

# Set Wine prefix to .wine in current directory
export WINEPREFIX="$SCRIPT_DIR/.wine"
export WINEDEBUG=-all

log_info "Working directory: $(pwd)"
log_info "Wine prefix: $WINEPREFIX"

# Initialize Wine prefix if it doesn't exist
if [ ! -d "$WINEPREFIX" ]; then
    log_info "Initializing new Wine prefix (this may take a while)..."
fi

log_info "Checking if Git for Windows is already installed..."
GIT_VERSION=$(wine git --version 2>/dev/null || GIT_VERSION="Not Installed")

if echo "$GIT_VERSION" | grep -qi "git version"; then
    log_info "Git is already installed: $GIT_VERSION"
    log_info "Skipping Git installation."
else
    log_warn "Git not found. Error: $GIT_VERSION"

    log_info "Fetching latest Git for Windows version..."
    GIT_URL=$(curl -s https://api.github.com/repos/git-for-windows/git/releases/latest \
        | grep browser_download_url \
        | grep '64-bit.exe"' \
        | sed -E 's/.*"([^"]+)".*/\1/')

    if [ -z "$GIT_URL" ]; then
        log_error "Failed to determine latest Git for Windows download URL."
        exit 1
    fi

    GIT_INSTALLER="${GIT_URL##*/}"
    log_info "Latest Git installer: $GIT_INSTALLER"
    log_info "URL: $GIT_URL"

    if [ ! -f "$GIT_INSTALLER" ]; then
        log_info "Downloading Git for Windows..."
        if ! curl -L -o "$GIT_INSTALLER" "$GIT_URL"; then
            log_error "Failed to download Git for Windows"
            exit 1
        fi
    fi

    log_info "Installing Git for Windows (this may take a while)..."
    if ! wine "$GIT_INSTALLER" /SILENT /NORESTART; then
        log_error "Git installation failed"
        exit 1
    fi

    rm "$GIT_INSTALLER"
    log_info "Git for Windows installed successfully."
fi

# Check if Python 3.12 is already installed
log_info "Checking if Python 3.12 is already installed..."
PYTHON_VERSION=$(wine python --version 2>/dev/null || PYTHON_VERSION="Not Installed")

if echo "$PYTHON_VERSION" | grep -q "Python 3.12"; then
    log_info "Python 3.12 is already installed: $PYTHON_VERSION"
    log_info "Skipping installation."
else
    log_warn "Python 3.12 not found. Error: $PYTHON_VERSION"
    log_info "Proceeding with installation..."
    
    # Download Python 3.12 installer
    PYTHON_URL="https://www.python.org/ftp/python/3.12.0/python-3.12.0-amd64.exe"
    PYTHON_INSTALLER="python-3.12-installer.exe"
    
    log_info "Downloading Python 3.12..."
    if ! curl -L -o "$PYTHON_INSTALLER" "$PYTHON_URL"; then
        log_error "Failed to download Python installer"
        exit 1
    fi

    # Install Python silently
    log_info "Installing Python 3.12 (this may take a while)..."
    winetricks -q win10
    if ! wine "$PYTHON_INSTALLER" /quiet InstallAllUsers=1 PrependPath=1; then
        log_error "Python installation failed"
        exit 1
    fi

    log_info "Python 3.12 installed successfully"

    # Clean up installer
    rm -f "$PYTHON_INSTALLER"
fi

# Copy required DLL files, wine should have these but it does not
cp -p "wine/icuuc64.dll" ".wine/drive_c/windows/system32/icuuc.dll"
cp -p "wine/icuuc32.dll" ".wine/drive_c/windows/syswow64/icuuc.dll"

# Copy src to BUILD_DIR before starting
log_info "Copying src to $BUILD_DIR..."
rm -rf "$BUILD_DIR"
rm -rf "$DEST_DIR"
mkdir -p "$DEST_DIR"
mkdir -p "$BUILD_DIR_PARENT"
cp -rp "src" "$BUILD_DIR"

rm -f $BUILD_DIR/deps/winetricks \
      $BUILD_DIR/deps/SLScheevo/SLScheevo
rm -rf $BUILD_DIR/tools

log_info "Building SLScheevo and ACCELA"
wine pip install -r requirements.txt
wine pyinstaller --clean --noconfirm --workpath "$BUILD_DIR-SLScheevo" --distpath "$BUILD_DIR\deps\SLScheevo" "$BUILD_DIR\deps\SLScheevo\SLScheevo.spec"
wine pyinstaller --clean \
                 --noconfirm \
                 --noconsole \
                 --distpath "$DEST_DIR/" \
                 --workpath "$BUILD_DIR-ACCELA" \
                 --upx-exclude "Cryptodome" \
                 --upx-exclude "Cryptodome.Cipher" \
                 --upx-exclude "Cryptodome/Cipher/_ARC4.abi3.so" \
                 --add-data "$BUILD_DIR/res:res" \
                 --add-data "$BUILD_DIR/deps:deps" \
                 --add-data "$BUILD_DIR/data:data" \
                 --name "ACCELA" \
                 --icon "$BUILD_DIR/res/logo/icon.ico" \
                 --onefile $BUILD_DIR/main.py



# Download Visual C++ Redistributable 2015-2022
log_info "Downloading Visual C++ Redistributable 2015-2022..."
curl -L -o "$DEST_DIR/vc_redist.x64.exe" "https://aka.ms/vc14/vc_redist.x64.exe"
curl -L -o "$DEST_DIR/vc_redist.x86.exe" "https://aka.ms/vc14/vc_redist.x86.exe"

log_info "Build complete! Executable is in $DEST_DIR"
rm -rf "$BUILD_DIR"
rm -rf "$BUILD_DIR-ACCELA"
rm -rf "$BUILD_DIR-SLScheevo"
