#!/usr/bin/env bash
cd "$(dirname "$(realpath "$0")")"

DEST_DIR="dist/linux_bin"
BUILD_DIR_PARENT="build"
BUILD_DIR="$BUILD_DIR_PARENT/linux_bin"

# Colors for output
RED="\033[0;31m"
GREEN="\033[0;32m"
YELLOW="\033[1;33m"
NC="\033[0m" # No Color

set +eu
# Check if notify-send exists globally
command -v notify-send &> /dev/null
NOTIFY_SEND_AVAILABLE=$?
set -eu

# Logging functions
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 "INFO" "$1" 2>/dev/null
    fi
    set -eu
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 -u normal "WARNING" "$1" 2>/dev/null
    fi
    set -eu
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 -u critical "ERROR" "$1" 2>/dev/null
    fi
    set -eu
}

# Copy src to BUILD_DIR before starting
log_info "Copying src to $BUILD_DIR..."
rm -rf "$BUILD_DIR"
rm -rf "$DEST_DIR"
mkdir -p "$DEST_DIR"
mkdir -p "$BUILD_DIR_PARENT"
cp -rp "src" "$BUILD_DIR"

rm -f $BUILD_DIR/deps/winetricks \
      $BUILD_DIR/deps/SLScheevo/SLScheevo.exe
rm -rf $BUILD_DIR/tools

# Check if Python 3 is available
if ! command -v python3 &>/dev/null; then
    log_error "python3 not found"
    exit 1
fi

# Parse command line arguments
SETUP_VENV=false
PYTHON_ARGS=()

for arg in "$@"; do
    if [ "$arg" = "--venv" ]; then
        SETUP_VENV=true
    else
        PYTHON_ARGS+=("$arg")
    fi
done

setup_venv() {
    # Create virtual environment
    python3 -m venv .venv

    # Activate virtual environment
    source .venv/bin/activate

    # Install requirements
    if [ -f "requirements.txt" ]; then
        pip install -r requirements.txt
    else
        log_warn "requirements.txt not found, skipping pip install"
    fi
}

if [ "$SETUP_VENV" = true ]; then
    log_info "Setting up virtual environment and installing dependencies"
    setup_venv
else
    # Check if virtual environment already exists and activate it if it does
    if [ -d ".venv" ] && [ -f ".venv/bin/activate" ]; then
        source .venv/bin/activate
    else
        log_warn "No virtual environment found, creating"
        setup_venv
    fi
fi

log_info "Building SLScheevo and ACCELA"
pyinstaller --clean --noconfirm --workpath "$BUILD_DIR-SLScheevo" --distpath "$BUILD_DIR/deps/SLScheevo" $BUILD_DIR/deps/SLScheevo/SLScheevo.spec
pyinstaller --clean \
            --noconfirm \
            --distpath "$DEST_DIR/bin" \
            --workpath "$BUILD_DIR-ACCELA" \
            --upx-exclude "Cryptodome" \
            --upx-exclude "Cryptodome.Cipher" \
            --upx-exclude "Cryptodome/Cipher/_ARC4.abi3.so" \
            --add-data "$BUILD_DIR/res:res" \
            --add-data "$BUILD_DIR/deps:deps" \
            --add-data "$BUILD_DIR/data:data" \
            --name "ACCELA" \
            --icon "$BUILD_DIR/res/logo/icon.ico" \
            --onefile $BUILD_DIR/main.py

cp -fp ACCELAINSTALL "$DEST_DIR/ACCELAINSTALL"
cp -fp ACCELAUNINSTALL "$DEST_DIR/ACCELAUNINSTALL"
cp -fp $BUILD_DIR/res/logo/accela.png "$DEST_DIR/bin"

log_info "Build complete! Executable is in $DEST_DIR"
rm -rf "$BUILD_DIR"
rm -rf "$BUILD_DIR-ACCELA"
rm -rf "$BUILD_DIR-SLScheevo"
