#!/usr/bin/env bash
cd "$(dirname "$(realpath "$0")")"

DEST_DIR="dist/linux_appimage"
BUILD_DIR_PARENT="build"
BUILD_DIR="$BUILD_DIR_PARENT/linux_appimage"
APPDIR="$BUILD_DIR/AppDir"

# Using a standalone Python build to avoid libpython shared library errors on other distros
PYTHON_RELEASE_TAG="20241016"
PYTHON_VERSION="3.13.0"
PYTHON_URL="https://github.com/indygreg/python-build-standalone/releases/download/${PYTHON_RELEASE_TAG}/cpython-${PYTHON_VERSION}+${PYTHON_RELEASE_TAG}-x86_64-unknown-linux-gnu-install_only.tar.gz"

# Colors for output
RED="\033[0;31m"
GREEN="\033[0;32m"
YELLOW="\033[1;33m"
NC="\033[0m" # No Color

set +eu
# Check if appimagetool is available
if ! command -v appimagetool &>/dev/null; then
    echo -e "${RED}[ERROR]${NC} appimagetool not found. Please install it first."
    exit 1
fi

# Check if notify-send exists globally
command -v notify-send &> /dev/null
NOTIFY_SEND_AVAILABLE=$?
set -eu

# Logging functions
log_info() {
    echo -e "${GREEN}[INFO]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 "INFO" "$1" 2>/dev/null
    fi
    set -eu
}

log_warn() {
    echo -e "${YELLOW}[WARN]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 -u normal "WARNING" "$1" 2>/dev/null
    fi
    set -eu
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
    set +eu
    if [ "$NOTIFY_SEND_AVAILABLE" -eq 0 ]; then
        notify-send -t 5000 -u critical "ERROR" "$1" 2>/dev/null
    fi
    set -eu
}

# Copy src to BUILD_DIR before starting
log_info "Copying src to $BUILD_DIR..."
rm -rf "$BUILD_DIR"
rm -rf "$DEST_DIR"
mkdir -p "$DEST_DIR"
mkdir -p "$BUILD_DIR_PARENT"
mkdir -p "$APPDIR/bin/src"
cp -rp src/* "$APPDIR/bin/src"

# Remove DB builder
rm -rf $APPDIR/bin/src/tools

# Create a portable python environment in the AppDir
log_info "Downloading and bundling Standalone Python ${PYTHON_VERSION}..."
mkdir -p "$APPDIR/bin/.venv"

# Download python tarball to a temporary location
# We need wget or curl. Checking for wget, falling back to curl if needed.
if command -v wget &> /dev/null; then
    wget -q --show-progress "$PYTHON_URL" -O python.tar.gz
else
    curl -L "$PYTHON_URL" -o python.tar.gz
fi

# Extract into the AppDir
tar -xzf python.tar.gz -C "$APPDIR/bin/.venv" --strip-components=1
rm python.tar.gz

# Define bundled paths
BUNDLED_PYTHON="$APPDIR/bin/.venv/bin/python3"
BUNDLED_PIP="$APPDIR/bin/.venv/bin/pip3"

# Install requirements
cd "$(dirname "$(realpath "$0")")"
if [ -f "requirements.txt" ]; then
    log_info "Installing dependencies from requirements.txt..."
    "$BUNDLED_PYTHON" -m pip install -r requirements.txt --no-warn-script-location
else
    log_warn "requirements.txt not found, skipping pip install"
fi

log_info "Creating run environment"
cd "$(dirname "$(realpath "$0")")"
cp run.sh "$APPDIR/bin/run.sh"
cp "$APPDIR/bin/src/res/logo/accela.png" "$APPDIR/accela.png"

# Desktop file
cp ACCELA.desktop "$APPDIR/"
sed -i \
  -e 's|^Exec=.*|Exec=run.sh|' \
  -e 's|^Icon=.*|Icon=accela|' \
  "$APPDIR/ACCELA.desktop"

# Create AppRun script
cat > "$APPDIR/AppRun" << 'EOF'
#!/bin/sh
HERE="$(dirname "$(readlink -f "$0")")"
cd "$HERE/bin"
export APPIMAGE=1
exec "./run.sh" "$@"
EOF

chmod +x "$APPDIR/bin/run.sh"
chmod +x "$APPDIR/AppRun"

mkdir -p "$DEST_DIR/bin"
appimagetool "$APPDIR" "$DEST_DIR/bin/ACCELA.AppImage"
cp -fp "$APPDIR/bin/src/res/logo/accela.png" "$DEST_DIR/bin"
cp -fp ACCELAINSTALL "$DEST_DIR/ACCELAINSTALL"
cp -fp ACCELAUNINSTALL "$DEST_DIR/ACCELAUNINSTALL"

log_info "AppImage build complete at: $DEST_DIR"

rm -rf "$BUILD_DIR"
