float rgb_2_saturation(vec3 rgb) {
	float minrgb = min(min(rgb.r, rgb.g), rgb.b);
	float maxrgb = max(max(rgb.r, rgb.g), rgb.b);
	
	return (max(maxrgb, 1e-10) - max(minrgb, 1e-10)) / max(maxrgb, 1e-2);
}

float rgb_2_yc(vec3 rgb) { // Converts RGB to a luminance proxy, here called YC. YC is ~ Y + K * Chroma.
	float ycRadiusWeight = 1.75;
	float r = rgb[0]; float g = rgb[1]; float b = rgb[2];
	float chroma = sqrt(b * (b - g) + g * (g - r) + r * (r - b));

	return (b + g + r + ycRadiusWeight * chroma) / 3.0;
}

vec3 XYZ_2_xyY( vec3 XYZ ) {
	float divisor = max(XYZ[0] + XYZ[1] + XYZ[2], 1e-10);
	
	vec3 xyY = XYZ.xyy;
	     xyY.rg = XYZ.rg / divisor;

	return xyY;
}

vec3 xyY_2_XYZ(vec3 xyY) {
	vec3 XYZ = vec3(0.0);
    
	     XYZ.r = xyY.r * xyY.b / max(xyY.g, 1e-10);
	     XYZ.g = xyY.b;  
	     XYZ.b = (1.0 - xyY.r - xyY.g) * xyY.b / max(xyY.g, 1e-10);

	return XYZ;
}


mat3 calc_sat_adjust_matrix(float sat, vec3 rgb2Y) {
    mat3 M;
    M[0][0] = (1.0 - sat) * rgb2Y[0] + sat;
    M[1][0] = (1.0 - sat) * rgb2Y[0];
    M[2][0] = (1.0 - sat) * rgb2Y[0];
    
    M[0][1] = (1.0 - sat) * rgb2Y[1];
    M[1][1] = (1.0 - sat) * rgb2Y[1] + sat;
    M[2][1] = (1.0 - sat) * rgb2Y[1];
    
    M[0][2] = (1.0 - sat) * rgb2Y[2];
    M[1][2] = (1.0 - sat) * rgb2Y[2];
    M[2][2] = (1.0 - sat) * rgb2Y[2] + sat;

    return M;
}

float cubic_basis_shaper(float x, float w) {
    const mat4 M = mat4(
        -1.0 / 6.0,  3.0 / 6.0, -3.0 / 6.0,  1.0 / 6.0,
         3.0 / 6.0, -6.0 / 6.0,  3.0 / 6.0,  0.0 / 6.0,
        -3.0 / 6.0,  0.0 / 6.0,  3.0 / 6.0,  0.0 / 6.0,
         1.0 / 6.0,  4.0 / 6.0,  1.0 / 6.0,  0.0 / 6.0
    );

    float knots[5] = float[5] (
        w * -0.5,
        w * -0.25,
        0.0,
        w *  0.25,
        w *  0.5
    );

    float y = 0;
    if((x > knots[0]) && (x < knots[4])) {
        float knot_coord = (x - knots[0]) * 4.0 / w;
        int j = int(knot_coord);
        float t = knot_coord - j;

        vec4 monomials = vec4(pow3(t), pow2(t), t, 1.0);

        switch(j) {
            case 3:  y = monomials[0] * M[0][0] + monomials[1] * M[1][0] + monomials[2] * M[2][0] + monomials[3] * M[3][0]; break;
            case 2:  y = monomials[0] * M[0][1] + monomials[1] * M[1][1] + monomials[2] * M[2][1] + monomials[3] * M[3][1]; break;
            case 1:  y = monomials[0] * M[0][2] + monomials[1] * M[1][2] + monomials[2] * M[2][2] + monomials[3] * M[3][2]; break;
            case 0:  y = monomials[0] * M[0][3] + monomials[1] * M[1][3] + monomials[2] * M[2][3] + monomials[3] * M[3][3]; break;
            default: y = 0.0; break;
        }
    }

    return y * 3.0 / 2.0;
}

float segmented_spline_c5_fwd(float x, const SegmentedSplineParams_c5 C){
    const int N_KNOTS_LOW = 4;
    const int N_KNOTS_HIGH = 4;

    float xCheck = x;
    if(xCheck <= 0.0) xCheck = pow(2, -14); 

    float logx = log10( xCheck);
    float logy;

    if(logx <= log10(C.minPoint.x)){ 
        logy = logx * C.slopeLow + ( log10(C.minPoint.y) - C.slopeLow * log10(C.minPoint.x) );
    } else if((logx > log10(C.minPoint.x)) && (logx < log10(C.midPoint.x))){
        float knot_coord = (N_KNOTS_LOW-1) * (logx-log10(C.minPoint.x))/(log10(C.midPoint.x)-log10(C.minPoint.x));

        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(C.coefsLow[ j], C.coefsLow[ j + 1], C.coefsLow[ j + 2]);
        vec3 monomials = vec3(t * t, t, 1);

        logy = dot(monomials, M * cf);
    } else if((logx >= log10(C.midPoint.x)) && (logx < log10(C.maxPoint.x))){
        float knot_coord = (N_KNOTS_HIGH-1) * (logx-log10(C.midPoint.x))/(log10(C.maxPoint.x)-log10(C.midPoint.x));

        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(C.coefsHigh[ j], C.coefsHigh[ j + 1], C.coefsHigh[ j + 2]); 
        vec3 monomials = vec3(t * t, t, 1);
        
        logy = dot(monomials, M * cf);
    } else {
        logy = logx * C.slopeHigh + ( log10(C.maxPoint.y) - C.slopeHigh * log10(C.maxPoint.x) );
    }

    return pow(10, logy);
}

float segmented_spline_c9_fwd(float x){
    const int N_KNOTS_LOW = 8;
    const int N_KNOTS_HIGH = 8;

    SegmentedSplineParams_c9 C = ODT_48nits;

    C.minPoint.x = segmented_spline_c5_fwd(C.minPoint.x, RRT_PARAMS);
    C.midPoint.x = segmented_spline_c5_fwd(C.midPoint.x, RRT_PARAMS);
    C.maxPoint.x = segmented_spline_c5_fwd(C.maxPoint.x, RRT_PARAMS);

    float xCheck = x;
    if(xCheck <= 0.0) xCheck = 1e-4;

    float logx = log10( xCheck);
    float logy;

    if(logx <= log10(C.minPoint.x)){ 
        logy = logx * C.slopeLow + ( log10(C.minPoint.y) - C.slopeLow * log10(C.minPoint.x) );
    } else if((logx > log10(C.minPoint.x)) && (logx < log10(C.midPoint.x))){
        float knot_coord = (N_KNOTS_LOW-1) * (logx-log10(C.minPoint.x))/(log10(C.midPoint.x)-log10(C.minPoint.x));

        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(C.coefsLow[ j], C.coefsLow[ j + 1], C.coefsLow[ j + 2]);
        vec3 monomials = vec3(t * t, t, 1);

        logy = dot(monomials, M * cf);
    } else if((logx >= log10(C.midPoint.x)) && (logx < log10(C.maxPoint.x))){
        float knot_coord = (N_KNOTS_HIGH-1) * (logx-log10(C.midPoint.x))/(log10(C.maxPoint.x)-log10(C.midPoint.x));

        int j = int(knot_coord);
        float t = knot_coord - j;

        vec3 cf = vec3(C.coefsHigh[ j], C.coefsHigh[ j + 1], C.coefsHigh[ j + 2]); 
        vec3 monomials = vec3(t * t, t, 1);
        
        logy = dot(monomials, M * cf);
    } else {
        logy = logx * C.slopeHigh + ( log10(C.maxPoint.y) - C.slopeHigh * log10(C.maxPoint.x) );
    }

    return pow(10, logy);
}

float glow_fwd(float ycIn, float glowGainIn, float glowMid) {
	float glowGainOut;

	if (ycIn <= 2.0 / 3.0 * glowMid) {
		glowGainOut = glowGainIn;
	} else if ( ycIn >= 2.0 * glowMid) {
		glowGainOut = 0;
	} else {
		glowGainOut = glowGainIn * (glowMid / ycIn - 0.5);
	}

	return glowGainOut;
}

float sigmoid_shaper(float x) { // Sigmoid function in the range 0 to 1 spanning -2 to +2.
	float t = max(1.0 - abs(0.5 * x), 0.0);
	float y = 1.0 + sign(x) * (1.0 - t * t);
	
	return 0.5 * y;
}

float rgb_2_hue(vec3 rgb) { // Returns a geometric hue angle in degrees (0-360) based on RGB values.
	float hue;
	if (rgb[0] == rgb[1] && rgb[1] == rgb[2]) { // For neutral colors, hue is undefined and the function will return a quiet NaN value.
		hue = 0;
	} else {
		hue = (180.0 / pi) * atan(2.0 * rgb[0] - rgb[1] - rgb[2], sqrt(3.0) * (rgb[1] - rgb[2])); // flip due to opengl spec compared to hlsl
	}

	if (hue < 0.0) 
		hue = hue + 360.0;

	return clamp(hue, 0.0, 360.0);
}

float center_hue( float hue, float centerH){
    float hueCentered = hue - centerH;

    if (hueCentered < -180) {
        hueCentered = hueCentered + 360;
    } else if (hueCentered > 180) {
        hueCentered = hueCentered - 360;
    }

    return hueCentered;
}