#!/usr/bin/env bash
set -e  # Exit on error

# Check if zenity is installed
if ! command -v zenity &> /dev/null; then
    echo "Error: zenity is required but not installed. Please install zenity to run this script."
    exit 1
fi

#Paths
PREFIXPATH=$HOME/.local/share/Steam/steamapps/compatdata

# Ask user if they want to change the path
if zenity --question --title="CompatData Path" --text="Current path: $PREFIXPATH\n\nDo you want to change where the game should be located?"; then
    # User said yes - get new path
    NEWPATH=$(zenity --entry --title="New CompatData Path" --text="Enter the new path for compatdata:" --entry-text="$PREFIXPATH")

    # If user entered something and didn't cancel, update the path
    if [ -n "$NEWPATH" ]; then
        PREFIXPATH="$NEWPATH"
    fi
fi

# Confirm final path
zenity --question --title="Confirm Path" --text="Final path: $PREFIXPATH\n\nIs this correct?" || {
    zenity --error --text="Operation cancelled."
    exit 1
}

# Dummy Script To Tell The Script Where It Will Be Operating In.
cd "$(dirname "$0")"
SCRIPT_DIR="$(pwd)"

# Detect which Protontricks command is available
detect_protontricks() {
    if command -v protontricks &> /dev/null; then
        echo "native"
    elif flatpak list | grep -q com.github.Matoking.protontricks; then
        echo "flatpak"
    else
        echo "none"
    fi
}

# This Function Will Use Protontricks to Find The Prefix ID of FFXISetup.exe.
findid() {
    local protontricks_type=$1
    local output=""

    if [ "$protontricks_type" = "flatpak" ]; then
        output=$(flatpak run com.github.Matoking.protontricks -s "FFXI.exe")
    else
        output=$(protontricks -s "FFXI.exe")
    fi

    local appid=$(echo "$output" | grep -oE '[0-9]+' | head -n 1)
    echo "Your Prefix ID"
    echo "Appid: $appid"
}

# Once The Prefix ID Is Shown User Would Enter The Appid Shown Above.
enterid(){
    read -rp "Enter Prefix ID: " prefixid
}

# Main script
PROTONTRICKS_TYPE=$(detect_protontricks)

case $PROTONTRICKS_TYPE in
    "native")
        echo "[Dotnet-Injector Native]"
        findid "native"
        ;;
    "flatpak")
        echo "[Dotnet-Injector Flatpak/Steamdeck]"
        findid "flatpak"
        ;;
    "none")
        echo "Error: Protontricks not found!"
        echo "Please install Protontricks either natively or via Flatpak"
        exit 1
        ;;
esac

sleep 1s
enterid

# Define paths
DEST="$PREFIXPATH/$prefixid/pfx"
BACKUP="$PREFIXPATH/$prefixid/pfx-original"

# If destination exists, move its contents to backup
if [ -d "$DEST" ]; then
    mkdir -p "$BACKUP"
    mv "$DEST/"* "$BACKUP/"
fi

# Move the contents of our pfx directory to the destination
mkdir -p "$DEST"

# Ask user with zenity dialog
action=$(zenity --question --title="File Operation" --text="Do you want to move the files?" \
          --ok-label="Move" --cancel-label="Copy" && echo "move" || echo "copy")

# Perform the operation
if [ "$action" = "move" ]; then
    mv "$SCRIPT_DIR/pfx/"* "$DEST/"
else
    cp -r "$SCRIPT_DIR/pfx/"* "$DEST/"
fi

notify-send "Prefix installed"
