#include <iostream>
#include <iomanip>
#include <cmath>
#include <algorithm>

class Picture
{
private:
  int **matrix;
  int width;
  int height;

public:
  /**
   * Default constructor. Creates an empty Picture object with
   * width and height set to 0, and matrix set to nullptr.
   */
  Picture() : matrix(nullptr), width(0), height(0) {}

  /**
   * Constructs a Picture object with the specified width and height.
   * Initializes a matrix of size height x width, with each element
   * set to 0.
   *
   * @param w Width of the picture
   * @param h Height of the picture
   */
  Picture(int w, int h) : width(w), height(h)
  {
    matrix = new int *[height];
    for (int i = 0; i < height; ++i)
    {
      matrix[i] = new int[width];
      for (int j = 0; j < width; ++j)
      {
        matrix[i][j] = 0;
      }
    }
  }

  /**
   * Destructor - Frees the memory allocated by the Picture object.
   */
  ~Picture()
  {
    for (int i = 0; i < height; ++i)
    {
      delete[] matrix[i];
    }
    delete[] matrix;
  }

  /**
   * Copy Constructor - Copies the contents of another Picture object into this one.
   *
   * @param other Picture object which is copied
   */
  Picture(const Picture &other) : width(other.width), height(other.height)
  {
    matrix = new int *[height];
    for (int i = 0; i < height; ++i)
    {
      matrix[i] = new int[width];
      for (int j = 0; j < width; ++j)
      {
        matrix[i][j] = other.matrix[i][j];
      }
    }
  }

  /**
   * Returns the dimensions of the picture as a pair of integers.
   * The first element is the width and the second element is the height.
   *
   * @return A pair containing the width and height of the picture.
   */
  inline std::pair<int, int> getDimensions() const
  {
    return {width, height};
  }

  /**
   * Increases the brightness of the picture by the given amount. The amount
   * to increase the brightness by is given as a parameter. The picture elements
   * are increased by this amount, and the result is clamped to the range 0-512.
   *
   * @param s Amount to increase the brightness by
   */
  void Brightness(int s)
  {
    for (int i = 0; i < height; ++i)
    {
      for (int j = 0; j < width; ++j)
      {
        matrix[i][j] = std::clamp(matrix[i][j] + s, 0, 512);
      }
    }
  }

  /**
   * Reads the picture elements from the standard input. The picture must be
   * entered as a sequence of three-digit numbers, each number representing
   * one element of the picture. The numbers are read row by row, and each row
   * is read from left to right. The input is validated to ensure that all
   * elements are in the range 0-512.
   */
  void inputPicture()
  {
    std::cout << "\nUnesite elemente slike (" << width << "x" << height << "):\n";
    for (int i = 0; i < height; ++i)
    {
      for (int j = 0; j < width; ++j)
      {
        std::cout << "Unesite element (0-512) za red " << i + 1 << ", kolonu " << j + 1 << ": ";
        std::cin >> matrix[i][j];
        matrix[i][j] = std::clamp(matrix[i][j], 0, 512); // Osiguravamo opseg 0-512
      }
    }
  }

  /**
   * Prints the picture to the standard output. Each element of the picture
   * is printed as a four-digit number, and each row is printed on a separate
   * line.
   */
  void displayPicture() const
  {
    for (int i = 0; i < height; ++i)
    {
      for (int j = 0; j < width; ++j)
      {
        std::cout << std::setw(4) << matrix[i][j] << " ";
      }
      std::cout << "\n";
    }
  }

  /**
   * Resizes the picture to the given dimensions. The new picture is created
   * by interpolating the old one using the nearest neighbor method.
   *
   * @param nWidth  New width of the picture
   * @param nHeight New height of the picture
   */
  void resize(int nWidth, int nHeight)
  {
    double nXFactor = static_cast<double>(width) / nWidth;
    double nYFactor = static_cast<double>(height) / nHeight;

    int **newMatrix = new int *[nHeight];
    for (int i = 0; i < nHeight; ++i)
    {
      newMatrix[i] = new int[nWidth];
      for (int j = 0; j < nWidth; ++j)
      {
        int srcX = static_cast<int>(std::floor(j * nXFactor));
        int srcY = static_cast<int>(std::floor(i * nYFactor));
        newMatrix[i][j] = matrix[srcY][srcX];
      }
    }

    for (int i = 0; i < height; ++i)
    {
      delete[] matrix[i];
    }
    delete[] matrix;

    matrix = newMatrix;
    width = nWidth;
    height = nHeight;
  }

  /**
   * Inverts the picture, i.e. all elements are subtracted from 512.
   */
  void invert()
  {
    for (int i = 0; i < height; ++i)
    {
      for (int j = 0; j < width; ++j)
      {
        matrix[i][j] = 512 - matrix[i][j];
      }
    }
  }
};

int main()
{
  // Create Picture object size 10x10
  Picture picture(10, 10);
  picture.inputPicture();
  std::cout << "Ocitana slika:\n";
  picture.displayPicture();

  // Invert
  picture.invert();
  std::cout << "Invertovana slika:\n";
  picture.displayPicture();

  // Decrease Brightness by 20
  picture.Brightness(-20);
  std::cout << "Slika sa manjom jakoscu svjetla:\n";
  picture.displayPicture();

  // Copy the picture to another
  Picture copiedPicture = picture;
  auto dimensions = copiedPicture.getDimensions();
  std::cout << "Dimenzije kopirane slike: " << dimensions.first << " x " << dimensions.second << "\n";
  copiedPicture.displayPicture();

  // Resize the copied picture to 15x15
  copiedPicture.resize(15, 15);
  std::cout << "Kopirana slika nakon resize-a:\n";
  copiedPicture.displayPicture();

  return 0;
}
