using System;
using System.Collections.ObjectModel;
using System.Reactive;
using System.Threading.Tasks;
using Avalonia.Threading;
using ReactiveUI;
using RestaurantApp.Models;
using RestaurantApp.Services;

namespace RestaurantApp.ViewModels;

public class ReservationsViewModel : ViewModelBase
{
    private ObservableCollection<Reservation> _reservations;
    public ObservableCollection<Reservation> Reservations
    {
        get => _reservations;
        set => this.RaiseAndSetIfChanged(ref _reservations, value);
    }
        
    private ObservableCollection<Customer> _customers;
    public ObservableCollection<Customer> Customers
    {
        get => _customers;
        set => this.RaiseAndSetIfChanged(ref _customers, value);
    }
        
    private Reservation _selectedReservation;
    public Reservation SelectedReservation
    {
        get => _selectedReservation;
        set => this.RaiseAndSetIfChanged(ref _selectedReservation, value);
    }
        
    public ReactiveCommand<Unit, Unit> AddCommand { get; }
    public ReactiveCommand<Unit, Unit> DeleteCommand { get; }
    public ReactiveCommand<Unit, Unit> RefreshCommand { get; }
    public ReactiveCommand<Unit, Unit> SaveCommand { get; }
        
    public ReservationsViewModel()
    {
        LoadData();
            
        AddCommand = ReactiveCommand.Create(AddReservation, outputScheduler: RxApp.MainThreadScheduler);
        DeleteCommand = ReactiveCommand.CreateFromTask(DeleteReservation, outputScheduler: RxApp.MainThreadScheduler);
        RefreshCommand = ReactiveCommand.Create(LoadData, outputScheduler: RxApp.MainThreadScheduler);
        SaveCommand = ReactiveCommand.CreateFromTask(SaveReservation, outputScheduler: RxApp.MainThreadScheduler);
    }
    
    private async Task SaveReservation()
    {
        if (SelectedReservation?.ReservationID == null)
        {
            await Utils.ShowError("Validation Error", "Order wasn't found, remember to click add new if you want to add");
            return;
        }
        
        if (SelectedReservation.ReservationID == 0)
        {
            DatabaseService.AddReservation(SelectedReservation);
        }
        else
        {
            DatabaseService.UpdateReservation(SelectedReservation);
        }
    
        LoadData();
    }
        
    private void LoadData()
    {
        Dispatcher.UIThread.InvokeAsync(() =>
        {
            Reservations = new ObservableCollection<Reservation>(DatabaseService.GetReservations());
            Customers = new ObservableCollection<Customer>(DatabaseService.GetCustomers());
        });
    }
        
    private void AddReservation()
    {
        SelectedReservation = new Reservation
        {
            ReservationDate = DateTime.Now
        };
    }
        
    private async Task DeleteReservation()
    {
        if (SelectedReservation == null) return;
            
        var confirm = await Utils.ShowConfirmation("Delete Reservation", 
            $"Delete reservation for {SelectedReservation.CustomerName}?");
            
        if (confirm)
        {
            DatabaseService.DeleteReservation(SelectedReservation.ReservationID);
            LoadData();
        }
    }
}