using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using System.Reactive;
using System.Threading.Tasks;
using Avalonia.Threading;
using ReactiveUI;
using RestaurantApp.Models;
using RestaurantApp.Services;

namespace RestaurantApp.ViewModels;

public class OrdersViewModel : ViewModelBase
{
    private ObservableCollection<Order> _orders = new();
    public ObservableCollection<Order> Orders
    {
        get => _orders;
        set => this.RaiseAndSetIfChanged(ref _orders, value);
    }
        
    private ObservableCollection<Customer> _customers = new();
    public ObservableCollection<Customer> Customers
    {
        get => _customers;
        set => this.RaiseAndSetIfChanged(ref _customers, value);
    }
        
    private ObservableCollection<MenuItem> _menuItems = new();
    public ObservableCollection<MenuItem> MenuItems
    {
        get => _menuItems;
        set => this.RaiseAndSetIfChanged(ref _menuItems, value);
    }
        
    private Order _selectedOrder = new();
    public Order SelectedOrder
    {
        get => _selectedOrder;
        set
        {
            this.RaiseAndSetIfChanged(ref _selectedOrder, value);
            if (value != null && value.OrderID != 0)
            {
                LoadOrderDetails(value.OrderID);
            }
        }
    }
        
    private OrderDetail _selectedOrderDetail = new();
    public OrderDetail SelectedOrderDetail
    {
        get => _selectedOrderDetail;
        set => this.RaiseAndSetIfChanged(ref _selectedOrderDetail, value);
    }
        
    private OrderDetail _newOrderItem = new();
    public OrderDetail NewOrderItem
    {
        get => _newOrderItem;
        set => this.RaiseAndSetIfChanged(ref _newOrderItem, value);
    }
        
    public ReactiveCommand<Unit, Unit> AddCommand { get; }
    public ReactiveCommand<Unit, Unit> DeleteOrderCommand { get; }
    public ReactiveCommand<Unit, Unit> AddItemCommand { get; }
    public ReactiveCommand<OrderDetail, Unit> DeleteItemCommand { get; }
    public ReactiveCommand<Unit, Unit> RefreshCommand { get; }
    public ReactiveCommand<Unit, Unit> SaveOrderCommand { get; }
        
    public OrdersViewModel()
    {
        LoadData();
            
        AddCommand = ReactiveCommand.Create(AddOrder, outputScheduler: RxApp.MainThreadScheduler);
        DeleteOrderCommand = ReactiveCommand.CreateFromTask(DeleteOrder, outputScheduler: RxApp.MainThreadScheduler);
        AddItemCommand = ReactiveCommand.CreateFromTask(AddItemToOrder, outputScheduler: RxApp.MainThreadScheduler);
        DeleteItemCommand = ReactiveCommand.Create<OrderDetail>(DeleteOrderItem, outputScheduler: RxApp.MainThreadScheduler);
        RefreshCommand = ReactiveCommand.Create(LoadData, outputScheduler: RxApp.MainThreadScheduler);
        SaveOrderCommand = ReactiveCommand.CreateFromTask(SaveOrder, outputScheduler: RxApp.MainThreadScheduler);
    }
        
    private void LoadData()
    {
        Dispatcher.UIThread.InvokeAsync(() =>
        {
            Orders = new ObservableCollection<Order>(DatabaseService.GetOrders());
            Customers = new ObservableCollection<Customer>(DatabaseService.GetCustomers());
            MenuItems = new ObservableCollection<MenuItem>(DatabaseService.GetMenuItems());
        });
    }
        
    private void LoadOrderDetails(int orderId)
    {
        SelectedOrder.OrderDetails = new List<OrderDetail>(
            DatabaseService.GetOrderDetails(orderId));
            
        // Calculate total
        SelectedOrder.TotalAmount = SelectedOrder.OrderDetails.Sum(od => od.Total);
        OnPropertyChanged(nameof(SelectedOrder));
    }
        
    private void AddOrder()
    {
        SelectedOrder = new Order
        {
            OrderDate = DateTime.Now,
            OrderDetails = []
        };
        NewOrderItem = new OrderDetail();
    }
        
    private async Task DeleteOrder()
    {
        if (SelectedOrder == null || SelectedOrder.OrderID == 0) return;
            
        var confirm = await Utils.ShowConfirmation("Delete Order", 
            $"Delete order #{SelectedOrder.OrderID}?");
            
        if (confirm)
        {
            DatabaseService.DeleteOrder(SelectedOrder.OrderID);
            LoadData();
        }
    }
        
    private async Task AddItemToOrder()
    {
        if (SelectedOrder == null || SelectedOrder.OrderID == 0)
        {
            await Utils.ShowError("Save Order First", "Please save the order before adding items");
            return;
        }
            
        if (NewOrderItem.MenuItemID == 0)
        {
            await Utils.ShowError("Selection Error", "Please select a menu item");
            return;
        }
            
        if (NewOrderItem.Quantity <= 0)
        {
            await Utils.ShowError("Quantity Error", "Quantity must be at least 1");
            return;
        }
            
        try
        {
            // Get menu item details
            var menuItem = MenuItems.First(m => m.MenuItemID == NewOrderItem.MenuItemID);
                
            // Create new order detail
            var newDetail = new OrderDetail
            {
                OrderID = SelectedOrder.OrderID,
                MenuItemID = NewOrderItem.MenuItemID,
                Quantity = NewOrderItem.Quantity,
                ItemName = menuItem.Name,
                Price = menuItem.Price
            };
                
            // Add to database
            newDetail.OrderDetailID = DatabaseService.AddOrderDetail(newDetail);
                
            // Add to local collection
            SelectedOrder.OrderDetails.Add(newDetail);
                
            // Update total
            SelectedOrder.TotalAmount += newDetail.Total;
            OnPropertyChanged(nameof(SelectedOrder));
                
            // Reset new item
            NewOrderItem = new OrderDetail();
        }
        catch (Exception ex)
        {
            await Utils.ShowError("Add Item Error", $"Could not add item: {ex.Message}");
        }
    }
        
    private void DeleteOrderItem(OrderDetail item)
    {
        if (item == null) return;
            
        // Remove from database
        DatabaseService.DeleteOrderDetail(item.OrderDetailID);
            
        // Remove from local collection
        SelectedOrder.OrderDetails.Remove(item);
            
        // Update total
        SelectedOrder.TotalAmount -= item.Total;
        OnPropertyChanged(nameof(SelectedOrder));
    }
        
    private async Task SaveOrder()
    {
        if (SelectedOrder?.OrderID == null)
        {
            await Utils.ShowError("Validation Error", "Order wasn't found, remember to click add new if you want to add");
            return;
        }
        
        if (SelectedOrder.CustomerID == 0)
        {
            await Utils.ShowError("Validation Error", "Please select a customer");
            return;
        }
            
        try
        {
            if (SelectedOrder.OrderID == 0)
            {
                // Add new order
                var newId = DatabaseService.AddOrder(SelectedOrder);
                SelectedOrder.OrderID = newId;
                Orders.Add(SelectedOrder);
            }
            else
            {
                // Update existing order
                DatabaseService.UpdateOrder(SelectedOrder);
                // Refresh item in collection
                var index = Orders.IndexOf(Orders.First(o => o.OrderID == SelectedOrder.OrderID));
                Orders[index] = SelectedOrder;
            }
                
            // Load details if any were added before save
            if (SelectedOrder.OrderDetails.Any())
            {
                LoadOrderDetails(SelectedOrder.OrderID);
            }
        }
        catch (Exception ex)
        {
            await Utils.ShowError("Save Error", $"Could not save order: {ex.Message}");
        }
    }
}