using System;
using System.Collections.ObjectModel;
using System.Linq;
using System.Reactive;
using System.Threading.Tasks;
using Avalonia.Threading;
using ReactiveUI;
using RestaurantApp.Models;
using RestaurantApp.Services;

namespace RestaurantApp.ViewModels;

public class MenuItemsViewModel : ViewModelBase
{
    private ObservableCollection<MenuItem> _menuItems = new();
    public ObservableCollection<MenuItem> MenuItems
    {
        get => _menuItems;
        set => this.RaiseAndSetIfChanged(ref _menuItems, value);
    }
        
    private MenuItem _selectedMenuItem = new();
    public MenuItem SelectedMenuItem
    {
        get => _selectedMenuItem;
        set => this.RaiseAndSetIfChanged(ref _selectedMenuItem, value);
    }
        
    public ReactiveCommand<Unit, Unit> AddCommand { get; }
    public ReactiveCommand<Unit, Unit> DeleteCommand { get; }
    public ReactiveCommand<Unit, Unit> RefreshCommand { get; }
    public ReactiveCommand<Unit, Unit> SaveCommand { get; }
        
    public MenuItemsViewModel()
    {
        LoadMenuItems();
            
        AddCommand = ReactiveCommand.Create(AddMenuItem, outputScheduler: RxApp.MainThreadScheduler);
        DeleteCommand = ReactiveCommand.CreateFromTask(DeleteMenuItem, outputScheduler: RxApp.MainThreadScheduler);
        RefreshCommand = ReactiveCommand.Create(LoadMenuItems, outputScheduler: RxApp.MainThreadScheduler);
        SaveCommand = ReactiveCommand.CreateFromTask(SaveMenuItem, outputScheduler: RxApp.MainThreadScheduler);
    }
        
    private void LoadMenuItems()
    {
        Dispatcher.UIThread.InvokeAsync(() =>
        {
            MenuItems = new ObservableCollection<MenuItem>(DatabaseService.GetMenuItems());
        });
    }
        
    private void AddMenuItem()
    {
        SelectedMenuItem = new MenuItem
        {
            MenuItemID = 0
        };
    }
        
    private async Task DeleteMenuItem()
    {
        if (SelectedMenuItem == null || SelectedMenuItem.MenuItemID == 0) return;
            
        var confirm = await Utils.ShowConfirmation("Delete Menu Item", 
            $"Delete {SelectedMenuItem.Name}?");
            
        if (confirm)
        {
            DatabaseService.DeleteMenuItem(SelectedMenuItem.MenuItemID);
            AddMenuItem();
            LoadMenuItems();
        }
    }
        
    private async Task SaveMenuItem()
    {
        if (SelectedMenuItem?.MenuItemID == null)
        {
            await Utils.ShowError("Validation Error", "Item wasn't found, remember to click add new if you want to add an item");
            return;
        }
        if (string.IsNullOrWhiteSpace(SelectedMenuItem.Name))
        {
            await Utils.ShowError("Validation Error", "Name is required");
            return;
        }
        if (string.IsNullOrWhiteSpace(SelectedMenuItem.Category))
        {
            await Utils.ShowError("Validation Error", "Category is required");
            return;
        }
        if (SelectedMenuItem.Price < 0)
        {
            await Utils.ShowError("Validation Error", "Price must be greater than or equal to 0");
            return;
        }
            
        try
        {
            if (DatabaseService.ExistsMenuItem(SelectedMenuItem))
            {
                // Update existing item
                DatabaseService.UpdateMenuItem(SelectedMenuItem);
                // Refresh item in collection
                var index = MenuItems.IndexOf(MenuItems.First(m => m.MenuItemID == SelectedMenuItem.MenuItemID));
                MenuItems[index] = SelectedMenuItem;
            }
            else
            {
                // Doesn't exist; Add new item
                var newId = DatabaseService.AddMenuItem(SelectedMenuItem);
                SelectedMenuItem.MenuItemID = newId;
                MenuItems.Add(SelectedMenuItem);
            }
        }
        catch (Exception ex)
        {
            await Utils.ShowError("Save Error", $"Could not save menu item: {ex.Message}");
        }
    }
}