﻿using System.Reactive;
using System.Threading.Tasks;
using Avalonia;
using Avalonia.Controls.ApplicationLifetimes;
using MsBox.Avalonia;
using MsBox.Avalonia.Enums;
using ReactiveUI;
using RestaurantApp.Views;

namespace RestaurantApp.ViewModels;

public class MainWindowViewModel : ViewModelBase
{
    private ViewModelBase _currentView;
    public ViewModelBase CurrentView
    {
        get => _currentView;
        set => this.RaiseAndSetIfChanged(ref _currentView, value);
    }
        
    // Navigation commands
    public ReactiveCommand<Unit, ViewModelBase> NavigateDashboardCommand { get; }
    public ReactiveCommand<Unit, ViewModelBase> NavigateCustomersCommand { get; }
    public ReactiveCommand<Unit, ViewModelBase> NavigateMenuItemsCommand { get; }
    public ReactiveCommand<Unit, ViewModelBase> NavigateOrdersCommand { get; }
    public ReactiveCommand<Unit, ViewModelBase> NavigateReservationsCommand { get; }
    public ReactiveCommand<Unit, Unit> LogoutCommand { get; }

    public MainWindowViewModel()
    {
        // Start with dashboard view
        CurrentView = new DashboardViewModel();
            
        // Initialize commands
        NavigateDashboardCommand = ReactiveCommand.Create(() => CurrentView = new DashboardViewModel(), outputScheduler: RxApp.MainThreadScheduler);
        NavigateCustomersCommand = ReactiveCommand.Create(() => CurrentView = new CustomersViewModel(), outputScheduler: RxApp.MainThreadScheduler);
        NavigateMenuItemsCommand = ReactiveCommand.Create(() => CurrentView = new MenuItemsViewModel(), outputScheduler: RxApp.MainThreadScheduler);
        NavigateOrdersCommand = ReactiveCommand.Create(() => CurrentView = new OrdersViewModel(), outputScheduler: RxApp.MainThreadScheduler);
        NavigateReservationsCommand = ReactiveCommand.Create(() => CurrentView = new ReservationsViewModel(), outputScheduler: RxApp.MainThreadScheduler);

        LogoutCommand = ReactiveCommand.CreateFromTask(Logout, outputScheduler: RxApp.MainThreadScheduler);
    }

    private async Task Logout()
    {
        var result = await ShowConfirmation("Logout", "Are you sure you want to logout?");
        if (result)
        {
            // Close current window and show login window
            var loginWindow = new LoginWindow();
            loginWindow.Show();
            (Application.Current.ApplicationLifetime as IClassicDesktopStyleApplicationLifetime)
                .MainWindow.Close();
            (Application.Current.ApplicationLifetime as IClassicDesktopStyleApplicationLifetime)
                .MainWindow = loginWindow;
        }
    }

    private async Task<bool> ShowConfirmation(string title, string message)
    {
        var box = MessageBoxManager.GetMessageBoxStandard(
            title,
            message,
            ButtonEnum.YesNo,
            Icon.Question
        );

        return await box.ShowAsync() == ButtonResult.Yes;
    }
}