using System;
using System.Collections.ObjectModel;
using System.Linq;
using System.Reactive;
using System.Threading.Tasks;
using Avalonia.Threading;
using ReactiveUI;
using RestaurantApp.Models;
using RestaurantApp.Services;

namespace RestaurantApp.ViewModels;

public class CustomersViewModel : ViewModelBase
{
    private ObservableCollection<Customer> _customers = new();
    public ObservableCollection<Customer> Customers
    {
        get => _customers;
        set => this.RaiseAndSetIfChanged(ref _customers, value);
    }
        
    private Customer _selectedCustomer = new();
    public Customer SelectedCustomer
    {
        get => _selectedCustomer;
        set => this.RaiseAndSetIfChanged(ref _selectedCustomer, value);
    }
        
    public ReactiveCommand<Unit, Unit> AddCommand { get; }
    public ReactiveCommand<Unit, Unit> EditCommand { get; }
    public ReactiveCommand<Unit, Unit> DeleteCommand { get; }
    public ReactiveCommand<Unit, Unit> RefreshCommand { get; }
    public ReactiveCommand<Unit, Unit> SaveCommand { get; }

    public CustomersViewModel()
    {
        LoadCustomers();
            
        AddCommand = ReactiveCommand.Create(AddCustomer, outputScheduler: RxApp.MainThreadScheduler);
        EditCommand = ReactiveCommand.Create(EditCustomer, outputScheduler: RxApp.MainThreadScheduler);
        DeleteCommand = ReactiveCommand.CreateFromTask(DeleteCustomer, outputScheduler: RxApp.MainThreadScheduler);
        RefreshCommand = ReactiveCommand.Create(LoadCustomers, outputScheduler: RxApp.MainThreadScheduler);
        SaveCommand = ReactiveCommand.CreateFromTask(SaveCustomer, outputScheduler: RxApp.MainThreadScheduler);
    }
        
    private void LoadCustomers()
    {
        Dispatcher.UIThread.InvokeAsync(() =>
        {
            Customers = new ObservableCollection<Customer>(DatabaseService.GetCustomers());
        });
    }
        
    private void AddCustomer()
    {
        SelectedCustomer = new Customer();
    }
        
    private void EditCustomer()
    {
        // Edit logic (selection handled by binding)
    }
        
    private async Task DeleteCustomer()
    {
        if (SelectedCustomer == null || SelectedCustomer.CustomerID == 0) return;
            
        var confirm = await Utils.ShowConfirmation("Delete Customer", 
            $"Delete {SelectedCustomer.FirstName} {SelectedCustomer.LastName}?");
            
        if (confirm)
        {
            DatabaseService.DeleteCustomer(SelectedCustomer.CustomerID);
            LoadCustomers();
        }
    }
        
    private async Task SaveCustomer()
    {
        if (SelectedCustomer?.CustomerID == null)
        {
            await Utils.ShowError("Validation Error", "Customer wasn't found, remember to click add new if you want to add");
            return;
        }
        
        if (string.IsNullOrWhiteSpace(SelectedCustomer.FirstName) || 
            string.IsNullOrWhiteSpace(SelectedCustomer.LastName))
        {
            await Utils.ShowError("Validation Error", "First and last name are required");
            return;
        }
            
        try
        {
            if (SelectedCustomer.CustomerID == 0)
            {
                // Add new customer
                var newId = DatabaseService.AddCustomer(SelectedCustomer);
                SelectedCustomer.CustomerID = newId;
                Customers.Add(SelectedCustomer);
            }
            else
            {
                // Update existing customer
                DatabaseService.UpdateCustomer(SelectedCustomer);
                // Refresh item in collection
                var index = Customers.IndexOf(Customers.First(c => c.CustomerID == SelectedCustomer.CustomerID));
                Customers[index] = SelectedCustomer;
            }
        }
        catch (Exception ex)
        {
            await Utils.ShowError("Save Error", $"Could not save customer: {ex.Message}");
        }
    }
}