using System;
using System.Collections.Generic;
using MySqlConnector;
using RestaurantApp.Models;

namespace RestaurantApp.Services;

public static class DatabaseService
{
    public static MySqlConnection GetConnection()
    {
        return new MySqlConnection(ConnectionString);
    }

    private const string ConnectionString = "Server=localhost;Database=Restoran;User ID=petar;Password=petaradmin;";

    public static int GetTotalCustomers()
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand("SELECT COUNT(*) FROM Customers", conn);
        return Convert.ToInt32(cmd.ExecuteScalar());
    }

    public static int GetUpcomingReservationsCount()
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            "SELECT COUNT(*) FROM Reservations WHERE ReservationDate >= CURDATE()",
            conn);
        return Convert.ToInt32(cmd.ExecuteScalar());
    }

    public static int GetTodaysOrdersCount()
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            "SELECT COUNT(*) FROM Orders WHERE OrderDate = CURDATE()",
            conn);
        return Convert.ToInt32(cmd.ExecuteScalar());
    }

    public static int GetMenuItemsCount()
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand("SELECT COUNT(*) FROM MenuItems", conn);
        return Convert.ToInt32(cmd.ExecuteScalar());
    }

    public static bool ValidateLogin(string username, string password)
    {
        using var conn = new MySqlConnection(ConnectionString);
        conn.Open();

        using var cmd = new MySqlCommand(
            "SELECT password FROM users WHERE username = @username",
            conn);

        cmd.Parameters.AddWithValue("@username", username);

        var hash = cmd.ExecuteScalar() as string;
        return hash != null && BCrypt.Net.BCrypt.Verify(password, hash);
    }

    public static List<Reservation> GetUpcomingReservations(int limit = 5)
    {
        var list = new List<Reservation>();
        using var conn = GetConnection();
        conn.Open();

        using var cmd = new MySqlCommand(@"
                SELECT r.ReservationID, r.ReservationDate, r.PartySize, 
                       CONCAT(c.FirstName, ' ', c.LastName) AS CustomerName
                FROM Reservations r
                JOIN Customers c ON r.CustomerID = c.CustomerID
                WHERE r.ReservationDate >= NOW()
                ORDER BY r.ReservationDate ASC
                LIMIT @limit", conn);

        cmd.Parameters.AddWithValue("@limit", limit);

        using var reader = cmd.ExecuteReader();
        while (reader.Read())
        {
            list.Add(new Reservation
            {
                ReservationID = reader.GetInt32("ReservationID"),
                ReservationDate = reader.GetDateTime("ReservationDate"),
                PartySize = reader.GetInt32("PartySize"),
                CustomerName = reader.GetString("CustomerName")
            });
        }

        return list;
    }

    public static List<OrderSummary> GetRecentOrders(int limit = 5)
    {
        var list = new List<OrderSummary>();
        using var conn = GetConnection();
        conn.Open();

        using var cmd = new MySqlCommand(@"
        SELECT o.OrderID, o.OrderDate, 
               CONCAT(c.FirstName, ' ', c.LastName) AS CustomerName,
               SUM(mi.Price * od.Quantity) AS TotalAmount
        FROM Orders o
        JOIN Customers c ON o.CustomerID = c.CustomerID
        JOIN OrderDetails od ON o.OrderID = od.OrderID
        JOIN MenuItems mi ON od.MenuItemID = mi.MenuItemID
        GROUP BY o.OrderID
        ORDER BY o.OrderDate DESC
        LIMIT @limit", conn);

        cmd.Parameters.AddWithValue("@limit", limit);

        using var reader = cmd.ExecuteReader();
        while (reader.Read())
        {
            list.Add(new OrderSummary
            {
                OrderID = reader.GetInt32("OrderID"),
                OrderDate = reader.GetDateTime("OrderDate"),
                CustomerName = reader.GetString("CustomerName"),
                TotalAmount = reader.GetDecimal("TotalAmount")
            });
        }

        return list;
    }

    public static List<Customer> GetCustomers()
    {
        var list = new List<Customer>();
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand("SELECT * FROM Customers", conn);
        using var reader = cmd.ExecuteReader();
        while (reader.Read())
        {
            list.Add(new Customer
            {
                CustomerID = reader.GetInt32("CustomerID"),
                FirstName = reader.GetString("FirstName"),
                LastName = reader.GetString("LastName")
            });
        }

        return list;
    }

    public static List<MenuItem> GetMenuItems()
    {
        var list = new List<MenuItem>();
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand("SELECT * FROM MenuItems", conn);
        using var reader = cmd.ExecuteReader();
        while (reader.Read())
        {
            list.Add(new MenuItem
            {
                MenuItemID = reader.GetInt32("MenuItemID"),
                Name = reader.GetString("Name"),
                Description = reader.GetString("Description"),
                Price = reader.GetDecimal("Price"),
                Category = reader.GetString("Category")
            });
        }

        return list;
    }

    public static List<Reservation> GetReservations()
    {
        var list = new List<Reservation>();
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(@"
        SELECT r.*, CONCAT(c.FirstName, ' ', c.LastName) AS CustomerName
        FROM Reservations r
        JOIN Customers c ON r.CustomerID = c.CustomerID
        ORDER BY r.ReservationDate DESC", conn);
        using var reader = cmd.ExecuteReader();
        while (reader.Read())
        {
            list.Add(new Reservation
            {
                ReservationID = reader.GetInt32("ReservationID"),
                CustomerID = reader.GetInt32("CustomerID"),
                ReservationDate = reader.GetDateTime("ReservationDate"),
                PartySize = reader.GetInt32("PartySize"),
                Notes = reader.GetString("Notes"),
                CustomerName = reader.GetString("CustomerName")
            });
        }

        return list;
    }

    public static List<Order> GetOrders()
    {
        var list = new List<Order>();
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(@"
        SELECT o.*, CONCAT(c.FirstName, ' ', c.LastName) AS CustomerName
        FROM Orders o
        JOIN Customers c ON o.CustomerID = c.CustomerID
        ORDER BY o.OrderDate DESC", conn);
        using var reader = cmd.ExecuteReader();
        while (reader.Read())
        {
            list.Add(new Order
            {
                OrderID = reader.GetInt32("OrderID"),
                CustomerID = reader.GetInt32("CustomerID"),
                OrderDate = reader.GetDateTime("OrderDate"),
                CustomerName = reader.GetString("CustomerName")
            });
        }

        return list;
    }

    public static List<OrderDetail> GetOrderDetails(int orderId)
    {
        var list = new List<OrderDetail>();
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(@"
        SELECT od.*, mi.Name AS ItemName, mi.Price
        FROM OrderDetails od
        JOIN MenuItems mi ON od.MenuItemID = mi.MenuItemID
        WHERE od.OrderID = @orderId", conn);
        cmd.Parameters.AddWithValue("@orderId", orderId);
        using var reader = cmd.ExecuteReader();
        while (reader.Read())
        {
            list.Add(new OrderDetail
            {
                OrderDetailID = reader.GetInt32("OrderDetailID"),
                OrderID = reader.GetInt32("OrderID"),
                MenuItemID = reader.GetInt32("MenuItemID"),
                Quantity = reader.GetInt32("Quantity"),
                ItemName = reader.GetString("ItemName"),
                Price = reader.GetDecimal("Price")
            });
        }

        return list;
    }

    // ======================
    // CUSTOMER OPERATIONS
    // ======================
    public static int AddCustomer(Customer customer)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            "INSERT INTO Customers (FirstName, LastName) VALUES (@firstName, @lastName); SELECT LAST_INSERT_ID();",
            conn);

        cmd.Parameters.AddWithValue("@firstName", customer.FirstName);
        cmd.Parameters.AddWithValue("@lastName", customer.LastName);

        return Convert.ToInt32(cmd.ExecuteScalar());
    }

    public static void UpdateCustomer(Customer customer)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            "UPDATE Customers SET FirstName = @firstName, LastName = @lastName WHERE CustomerID = @id",
            conn);

        cmd.Parameters.AddWithValue("@firstName", customer.FirstName);
        cmd.Parameters.AddWithValue("@lastName", customer.LastName);
        cmd.Parameters.AddWithValue("@id", customer.CustomerID);

        cmd.ExecuteNonQuery();
    }

    public static void DeleteCustomer(int customerId)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            "DELETE FROM Customers WHERE CustomerID = @id",
            conn);

        cmd.Parameters.AddWithValue("@id", customerId);
        cmd.ExecuteNonQuery();
    }

    // ======================
    // MENU ITEM OPERATIONS
    // ======================
    public static int AddMenuItem(MenuItem item)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            @"INSERT INTO MenuItems (Name, Description, Price, Category) 
              VALUES (@name, @desc, @price, @category); 
              SELECT LAST_INSERT_ID();",
            conn);

        cmd.Parameters.AddWithValue("@name", item.Name);
        cmd.Parameters.AddWithValue("@desc", item.Description);
        cmd.Parameters.AddWithValue("@price", item.Price);
        cmd.Parameters.AddWithValue("@category", item.Category);

        return Convert.ToInt32(cmd.ExecuteScalar());
    }

    public static void UpdateMenuItem(MenuItem item)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            @"UPDATE MenuItems 
              SET Name = @name, Description = @desc, Price = @price, Category = @category
              WHERE MenuItemID = @id",
            conn);

        cmd.Parameters.AddWithValue("@name", item.Name);
        cmd.Parameters.AddWithValue("@desc", item.Description);
        cmd.Parameters.AddWithValue("@price", item.Price);
        cmd.Parameters.AddWithValue("@category", item.Category);
        cmd.Parameters.AddWithValue("@id", item.MenuItemID);

        cmd.ExecuteNonQuery();
    }

    public static bool ExistsMenuItem(MenuItem item)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(@"SELECT * FROM MenuItems WHERE MenuItemID = @id", conn);
        cmd.Parameters.AddWithValue("@name", item.Name);
        cmd.Parameters.AddWithValue("@desc", item.Description);
        cmd.Parameters.AddWithValue("@price", item.Price);
        cmd.Parameters.AddWithValue("@category", item.Category);
        cmd.Parameters.AddWithValue("@id", item.MenuItemID);
        int check = cmd.ExecuteNonQuery();
        if (check != 0)
        {
            return true;
        }

        return false;
    }

    public static void DeleteMenuItem(int menuItemId)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            "DELETE FROM MenuItems WHERE MenuItemID = @id",
            conn);

        cmd.Parameters.AddWithValue("@id", menuItemId);
        cmd.ExecuteNonQuery();
    }

    // ======================
    // RESERVATION OPERATIONS
    // ======================
    public static int AddReservation(Reservation reservation)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            @"INSERT INTO Reservations (CustomerID, ReservationDate, PartySize, Notes) 
              VALUES (@customerId, @date, @size, @notes); 
              SELECT LAST_INSERT_ID();",
            conn);

        cmd.Parameters.AddWithValue("@customerId", reservation.CustomerID);
        cmd.Parameters.AddWithValue("@date", reservation.ReservationDate);
        cmd.Parameters.AddWithValue("@size", reservation.PartySize);
        cmd.Parameters.AddWithValue("@notes", reservation.Notes ?? "");

        return Convert.ToInt32(cmd.ExecuteScalar());
    }

    public static void UpdateReservation(Reservation reservation)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            @"UPDATE Reservations 
              SET CustomerID = @customerId, 
                  ReservationDate = @date, 
                  PartySize = @size, 
                  Notes = @notes
              WHERE ReservationID = @id",
            conn);

        cmd.Parameters.AddWithValue("@customerId", reservation.CustomerID);
        cmd.Parameters.AddWithValue("@date", reservation.ReservationDate);
        cmd.Parameters.AddWithValue("@size", reservation.PartySize);
        cmd.Parameters.AddWithValue("@notes", reservation.Notes ?? "");
        cmd.Parameters.AddWithValue("@id", reservation.ReservationID);

        cmd.ExecuteNonQuery();
    }

    public static void DeleteReservation(int reservationId)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            "DELETE FROM Reservations WHERE ReservationID = @id",
            conn);

        cmd.Parameters.AddWithValue("@id", reservationId);
        cmd.ExecuteNonQuery();
    }

    // ======================
    // ORDER OPERATIONS
    // ======================
    public static int AddOrder(Order order)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            @"INSERT INTO Orders (CustomerID, OrderDate) 
              VALUES (@customerId, @date); 
              SELECT LAST_INSERT_ID();",
            conn);

        cmd.Parameters.AddWithValue("@customerId", order.CustomerID);
        cmd.Parameters.AddWithValue("@date", order.OrderDate);

        return Convert.ToInt32(cmd.ExecuteScalar());
    }

    public static void UpdateOrder(Order order)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            @"UPDATE Orders 
              SET CustomerID = @customerId, 
                  OrderDate = @date
              WHERE OrderID = @id",
            conn);

        cmd.Parameters.AddWithValue("@customerId", order.CustomerID);
        cmd.Parameters.AddWithValue("@date", order.OrderDate);
        cmd.Parameters.AddWithValue("@id", order.OrderID);

        cmd.ExecuteNonQuery();
    }

    public static void DeleteOrder(int orderId)
    {
        using var conn = GetConnection();
        conn.Open();

        // First delete order details
        using var cmdDetails = new MySqlCommand(
            "DELETE FROM OrderDetails WHERE OrderID = @id",
            conn);
        cmdDetails.Parameters.AddWithValue("@id", orderId);
        cmdDetails.ExecuteNonQuery();

        // Then delete the order
        using var cmdOrder = new MySqlCommand(
            "DELETE FROM Orders WHERE OrderID = @id",
            conn);
        cmdOrder.Parameters.AddWithValue("@id", orderId);
        cmdOrder.ExecuteNonQuery();
    }

    // ======================
    // ORDER DETAIL OPERATIONS
    // ======================
    public static int AddOrderDetail(OrderDetail detail)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            @"INSERT INTO OrderDetails (OrderID, MenuItemID, Quantity) 
              VALUES (@orderId, @itemId, @qty); 
              SELECT LAST_INSERT_ID();",
            conn);

        cmd.Parameters.AddWithValue("@orderId", detail.OrderID);
        cmd.Parameters.AddWithValue("@itemId", detail.MenuItemID);
        cmd.Parameters.AddWithValue("@qty", detail.Quantity);

        return Convert.ToInt32(cmd.ExecuteScalar());
    }

    public static void UpdateOrderDetail(OrderDetail detail)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            @"UPDATE OrderDetails 
              SET Quantity = @qty
              WHERE OrderDetailID = @id",
            conn);

        cmd.Parameters.AddWithValue("@qty", detail.Quantity);
        cmd.Parameters.AddWithValue("@id", detail.OrderDetailID);

        cmd.ExecuteNonQuery();
    }

    public static void DeleteOrderDetail(int detailId)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            "DELETE FROM OrderDetails WHERE OrderDetailID = @id",
            conn);

        cmd.Parameters.AddWithValue("@id", detailId);
        cmd.ExecuteNonQuery();
    }

    public static void DeleteOrderItem(int orderDetailId)
    {
        using var conn = GetConnection();
        conn.Open();
        using var cmd = new MySqlCommand(
            "DELETE FROM OrderDetails WHERE OrderDetailID = @id",
            conn);

        cmd.Parameters.AddWithValue("@id", orderDetailId);
        cmd.ExecuteNonQuery();
    }
}