using System;
using System.Collections.Generic;
using System.Linq;
using Menu;
using Menu.Remix.MixedUI;
using UnityEngine;

namespace RainGradient;

public partial class RainGradient
{
    public class RainGradientOptions : OptionInterface
    {
        private bool _init;
        public int CurTab;
        public readonly Configurable<int> stopCount;
        private List<OpTab> dynamicTabs = new();
        private readonly List<Configurable<Color>> dynamicColors = new();

        public RainGradientOptions()
        {
            // RainGradient tab
            stopCount = config.Bind(nameof(stopCount), 1, new ConfigurableInfo("How many stops to have.", null, "", "Input for amount of stops"));
        }

        public override void Initialize()
        {
            try
            {
                base.Initialize();

                Tabs = new OpTab[] { new(this, "RainGradient") };

                CurTab = 0;
                float x = 80f;
                float y = 540f;
                float sepr = 30f;

                AddTitle();
                AddText("Dynamic tab system for stops.", new Vector2(150f, y -= sepr));
                AddUpDown(stopCount, new Vector2(x, y -= sepr), 150f);

                InitializeDynamicTabs();

                Log("Added all options...");
                _init = true;
            }
            catch (Exception ex)
            {
                LogError(ex, "Error opening RainGradient Options Menu");
            }
        }

        public override void Update()
        {
            if (!_init) return;
            
            // Check if the number of tabs needs to be updated
            if (dynamicTabs.Count != stopCount.Value)
            {
                InitializeDynamicTabs();
            }
        }

        private void InitializeDynamicTabs()
        {
            // Clear existing dynamic tabs
            Tabs = new []{ Tabs[0] };
            dynamicTabs.Clear();
            dynamicColors.Clear();

            // Recreate tabs based on stopCount value
            for (int i = 0; i < stopCount.Value; i++)
            {
                CurTab++;
                AddDynamicTab();
            }
        }

        private void AddDynamicTab()
        {
            var tab = new OpTab(this, $"Stop {CurTab}");
            Tabs = Tabs.Append(tab).ToArray();
            dynamicTabs.Add(tab);

            // Create a new configurable color for this tab
            var colorConfig = config.Bind($"StopColor{CurTab}", Color.white, new ConfigurableInfo($"Pick a color for Stop {CurTab}."));
            dynamicColors.Add(colorConfig);

            AddDynamicTabContent(colorConfig);
        }

        private void AddDynamicTabContent(Configurable<Color> colorConfig)
        {
            float x = 100f;
            float y = 540f;
            float sepr = 30f;

            AddTitle();
            AddColorPicker(colorConfig, new Vector2(x, y -= sepr));
        }

        private void AddTitle(string titleText = "RainGradient Options")
        {
            OpLabel title = new(new Vector2(150f, 560f), new Vector2(300f, 30f), titleText, bigText: true);
            OpLabel version = new(new Vector2(150f, 540f), new Vector2(300f, 30f), $"Version {PluginInfo.PluginVersion}");

            Tabs[CurTab].AddItems(title, version);
        }

        private void AddUpDown(Configurable<int> option, Vector2 pos, float size, int min = 1, int max = 10)
        {
            OpUpdown input = new(option, pos, size) { description = option.info.description };
            OpLabel label = new(pos.x + 40f, pos.y + 2f, option.info.Tags[0] as string) { description = option.info.description };

            Tabs[CurTab].AddItems(input, label);
        }

        private void AddText(string text, Vector2 pos, Color? clr = null)
        {
            clr ??= MenuColorEffect.rgbMediumGrey;

            OpLabel label = new(pos, new Vector2(300f, 30f), text)
            {
                color = (Color)clr,
            };

            Tabs[CurTab].AddItems(label);
        }

        private void AddColorPicker(Configurable<Color> config, Vector2 position)
        {
            OpColorPicker colorPicker = new(config, position);
            Tabs[CurTab].AddItems(colorPicker);
        }

        private void Log(string message)
        {
            Debug.Log($"[RainGradientOptions] {message}");
        }

        private void LogError(Exception ex, string message)
        {
            Debug.LogError($"[RainGradientOptions] {message}\n{ex}");
        }
    }
}