using System;
using System.Text;

namespace Epic.OnlineServices.P2P;

public struct SocketId
{
	public static readonly SocketId Empty = default(SocketId);

	private const int MaxSocketNameLength = 32;

	private const int ApiVersionLength = 4;

	private const int NullTerminatorSpace = 1;

	private const int TotalSizeInBytes = 37;

	private bool m_CacheValid;

	private string m_CachedSocketName;

	internal byte[] m_AllBytes;

	internal byte[] m_SwapBuffer;

	public string SocketName
	{
		get
		{
			if (m_CacheValid)
			{
				return m_CachedSocketName;
			}
			if (m_AllBytes == null)
			{
				return null;
			}
			RebuildStringFromBuffer();
			return m_CachedSocketName;
		}
		set
		{
			m_CachedSocketName = value;
			if (value == null)
			{
				m_CacheValid = true;
				return;
			}
			EnsureStorage();
			int num = Math.Min(32, value.Length);
			Encoding.ASCII.GetBytes(value, 0, num, m_AllBytes, 4);
			m_AllBytes[num + 4] = 0;
			m_CacheValid = true;
		}
	}

	internal void Set(ref SocketIdInternal other)
	{
		SocketName = other.SocketName;
	}

	internal bool PrepareForUpdate()
	{
		bool cacheValid = m_CacheValid;
		m_CacheValid = false;
		EnsureStorage();
		CopyIdToSwapBuffer();
		return cacheValid;
	}

	internal void CheckIfChanged(bool wasCacheValid)
	{
		if (!wasCacheValid || m_SwapBuffer == null || m_AllBytes == null)
		{
			return;
		}
		bool flag = true;
		for (int i = 0; i < m_SwapBuffer.Length; i++)
		{
			if (m_AllBytes[4 + i] != m_SwapBuffer[i])
			{
				flag = false;
				break;
			}
		}
		if (flag)
		{
			m_CacheValid = true;
		}
	}

	private void RebuildStringFromBuffer()
	{
		EnsureStorage();
		int i;
		for (i = 4; i < m_AllBytes.Length && m_AllBytes[i] != 0; i++)
		{
		}
		m_CachedSocketName = Encoding.ASCII.GetString(m_AllBytes, 4, i - 4);
		m_CacheValid = true;
	}

	private void EnsureStorage()
	{
		if (m_AllBytes == null || m_AllBytes.Length < 37)
		{
			m_AllBytes = new byte[37];
			m_SwapBuffer = new byte[33];
			Array.Copy(BitConverter.GetBytes(1), 0, m_AllBytes, 0, 4);
		}
	}

	private void CopyIdToSwapBuffer()
	{
		Array.Copy(m_AllBytes, 4, m_SwapBuffer, 0, m_SwapBuffer.Length);
	}
}
